/// PVE API client
/// Note that the following API URLs are not handled currently:
///
/// ```text
/// - /access
/// - /access/acl
/// - /access/domains/{realm}
/// - /access/domains/{realm}/sync
/// - /access/groups
/// - /access/groups/{groupid}
/// - /access/openid
/// - /access/openid/auth-url
/// - /access/openid/login
/// - /access/password
/// - /access/permissions
/// - /access/roles
/// - /access/roles/{roleid}
/// - /access/tfa
/// - /access/tfa/{userid}
/// - /access/tfa/{userid}/{id}
/// - /access/ticket
/// - /access/users
/// - /access/users/{userid}
/// - /access/users/{userid}/tfa
/// - /access/users/{userid}/token
/// - /access/users/{userid}/unlock-tfa
/// - /access/vncticket
/// - /cluster
/// - /cluster/acme
/// - /cluster/acme/account
/// - /cluster/acme/account/{name}
/// - /cluster/acme/challenge-schema
/// - /cluster/acme/directories
/// - /cluster/acme/meta
/// - /cluster/acme/plugins
/// - /cluster/acme/plugins/{id}
/// - /cluster/acme/tos
/// - /cluster/backup
/// - /cluster/backup-info
/// - /cluster/backup-info/not-backed-up
/// - /cluster/backup/{id}
/// - /cluster/backup/{id}/included_volumes
/// - /cluster/bulk-action
/// - /cluster/bulk-action/guest
/// - /cluster/bulk-action/guest/migrate
/// - /cluster/bulk-action/guest/shutdown
/// - /cluster/bulk-action/guest/start
/// - /cluster/bulk-action/guest/suspend
/// - /cluster/ceph
/// - /cluster/ceph/flags
/// - /cluster/ceph/flags/{flag}
/// - /cluster/ceph/metadata
/// - /cluster/ceph/status
/// - /cluster/config
/// - /cluster/config/apiversion
/// - /cluster/config/nodes
/// - /cluster/config/nodes/{node}
/// - /cluster/config/qdevice
/// - /cluster/config/totem
/// - /cluster/firewall
/// - /cluster/firewall/groups
/// - /cluster/firewall/groups/{group}
/// - /cluster/firewall/groups/{group}/{pos}
/// - /cluster/firewall/rules/{pos}
/// - /cluster/ha
/// - /cluster/ha/groups
/// - /cluster/ha/groups/{group}
/// - /cluster/ha/resources
/// - /cluster/ha/resources/{sid}
/// - /cluster/ha/resources/{sid}/migrate
/// - /cluster/ha/resources/{sid}/relocate
/// - /cluster/ha/rules
/// - /cluster/ha/rules/{rule}
/// - /cluster/ha/status
/// - /cluster/ha/status/current
/// - /cluster/ha/status/manager_status
/// - /cluster/jobs
/// - /cluster/jobs/realm-sync
/// - /cluster/jobs/realm-sync/{id}
/// - /cluster/jobs/schedule-analyze
/// - /cluster/log
/// - /cluster/mapping
/// - /cluster/mapping/dir
/// - /cluster/mapping/dir/{id}
/// - /cluster/mapping/pci
/// - /cluster/mapping/pci/{id}
/// - /cluster/mapping/usb
/// - /cluster/mapping/usb/{id}
/// - /cluster/metrics
/// - /cluster/metrics/server
/// - /cluster/metrics/server/{id}
/// - /cluster/nextid
/// - /cluster/notifications
/// - /cluster/notifications/endpoints
/// - /cluster/notifications/endpoints/gotify
/// - /cluster/notifications/endpoints/gotify/{name}
/// - /cluster/notifications/endpoints/sendmail
/// - /cluster/notifications/endpoints/sendmail/{name}
/// - /cluster/notifications/endpoints/smtp
/// - /cluster/notifications/endpoints/smtp/{name}
/// - /cluster/notifications/endpoints/webhook
/// - /cluster/notifications/endpoints/webhook/{name}
/// - /cluster/notifications/matcher-field-values
/// - /cluster/notifications/matcher-fields
/// - /cluster/notifications/matchers
/// - /cluster/notifications/matchers/{name}
/// - /cluster/notifications/targets
/// - /cluster/notifications/targets/{name}
/// - /cluster/notifications/targets/{name}/test
/// - /cluster/replication
/// - /cluster/replication/{id}
/// - /cluster/sdn/controllers/{controller}
/// - /cluster/sdn/dns
/// - /cluster/sdn/dns/{dns}
/// - /cluster/sdn/fabrics
/// - /cluster/sdn/fabrics/all
/// - /cluster/sdn/fabrics/fabric
/// - /cluster/sdn/fabrics/fabric/{id}
/// - /cluster/sdn/fabrics/node
/// - /cluster/sdn/fabrics/node/{fabric_id}
/// - /cluster/sdn/fabrics/node/{fabric_id}/{node_id}
/// - /cluster/sdn/ipams
/// - /cluster/sdn/ipams/{ipam}
/// - /cluster/sdn/ipams/{ipam}/status
/// - /cluster/sdn/vnets/{vnet}
/// - /cluster/sdn/vnets/{vnet}/firewall
/// - /cluster/sdn/vnets/{vnet}/firewall/options
/// - /cluster/sdn/vnets/{vnet}/firewall/rules
/// - /cluster/sdn/vnets/{vnet}/firewall/rules/{pos}
/// - /cluster/sdn/vnets/{vnet}/ips
/// - /cluster/sdn/vnets/{vnet}/subnets
/// - /cluster/sdn/vnets/{vnet}/subnets/{subnet}
/// - /cluster/sdn/zones/{zone}
/// - /cluster/tasks
/// - /nodes/{node}
/// - /nodes/{node}/aplinfo
/// - /nodes/{node}/apt
/// - /nodes/{node}/capabilities
/// - /nodes/{node}/capabilities/qemu
/// - /nodes/{node}/capabilities/qemu/cpu-flags
/// - /nodes/{node}/capabilities/qemu/machines
/// - /nodes/{node}/capabilities/qemu/migration
/// - /nodes/{node}/ceph
/// - /nodes/{node}/ceph/cfg
/// - /nodes/{node}/ceph/cfg/db
/// - /nodes/{node}/ceph/cfg/raw
/// - /nodes/{node}/ceph/cfg/value
/// - /nodes/{node}/ceph/cmd-safety
/// - /nodes/{node}/ceph/crush
/// - /nodes/{node}/ceph/fs
/// - /nodes/{node}/ceph/fs/{name}
/// - /nodes/{node}/ceph/init
/// - /nodes/{node}/ceph/log
/// - /nodes/{node}/ceph/mds
/// - /nodes/{node}/ceph/mds/{name}
/// - /nodes/{node}/ceph/mgr
/// - /nodes/{node}/ceph/mgr/{id}
/// - /nodes/{node}/ceph/mon
/// - /nodes/{node}/ceph/mon/{monid}
/// - /nodes/{node}/ceph/osd
/// - /nodes/{node}/ceph/osd/{osdid}
/// - /nodes/{node}/ceph/osd/{osdid}/in
/// - /nodes/{node}/ceph/osd/{osdid}/lv-info
/// - /nodes/{node}/ceph/osd/{osdid}/metadata
/// - /nodes/{node}/ceph/osd/{osdid}/out
/// - /nodes/{node}/ceph/osd/{osdid}/scrub
/// - /nodes/{node}/ceph/pool
/// - /nodes/{node}/ceph/pool/{name}
/// - /nodes/{node}/ceph/pool/{name}/status
/// - /nodes/{node}/ceph/restart
/// - /nodes/{node}/ceph/rules
/// - /nodes/{node}/ceph/start
/// - /nodes/{node}/ceph/status
/// - /nodes/{node}/ceph/stop
/// - /nodes/{node}/certificates
/// - /nodes/{node}/certificates/acme
/// - /nodes/{node}/certificates/acme/certificate
/// - /nodes/{node}/certificates/custom
/// - /nodes/{node}/certificates/info
/// - /nodes/{node}/disks
/// - /nodes/{node}/disks/directory
/// - /nodes/{node}/disks/directory/{name}
/// - /nodes/{node}/disks/initgpt
/// - /nodes/{node}/disks/list
/// - /nodes/{node}/disks/lvm
/// - /nodes/{node}/disks/lvm/{name}
/// - /nodes/{node}/disks/lvmthin
/// - /nodes/{node}/disks/lvmthin/{name}
/// - /nodes/{node}/disks/smart
/// - /nodes/{node}/disks/wipedisk
/// - /nodes/{node}/disks/zfs
/// - /nodes/{node}/disks/zfs/{name}
/// - /nodes/{node}/dns
/// - /nodes/{node}/execute
/// - /nodes/{node}/firewall
/// - /nodes/{node}/firewall/rules/{pos}
/// - /nodes/{node}/hardware
/// - /nodes/{node}/hardware/pci
/// - /nodes/{node}/hardware/pci/{pci-id-or-mapping}
/// - /nodes/{node}/hardware/pci/{pci-id-or-mapping}/mdev
/// - /nodes/{node}/hardware/usb
/// - /nodes/{node}/hosts
/// - /nodes/{node}/journal
/// - /nodes/{node}/lxc/{vmid}
/// - /nodes/{node}/lxc/{vmid}/clone
/// - /nodes/{node}/lxc/{vmid}/feature
/// - /nodes/{node}/lxc/{vmid}/firewall
/// - /nodes/{node}/lxc/{vmid}/firewall/rules/{pos}
/// - /nodes/{node}/lxc/{vmid}/interfaces
/// - /nodes/{node}/lxc/{vmid}/mtunnel
/// - /nodes/{node}/lxc/{vmid}/mtunnelwebsocket
/// - /nodes/{node}/lxc/{vmid}/rrd
/// - /nodes/{node}/lxc/{vmid}/rrddata
/// - /nodes/{node}/lxc/{vmid}/snapshot
/// - /nodes/{node}/lxc/{vmid}/snapshot/{snapname}
/// - /nodes/{node}/lxc/{vmid}/snapshot/{snapname}/config
/// - /nodes/{node}/lxc/{vmid}/snapshot/{snapname}/rollback
/// - /nodes/{node}/lxc/{vmid}/spiceproxy
/// - /nodes/{node}/lxc/{vmid}/status
/// - /nodes/{node}/lxc/{vmid}/status/reboot
/// - /nodes/{node}/lxc/{vmid}/status/resume
/// - /nodes/{node}/lxc/{vmid}/status/suspend
/// - /nodes/{node}/lxc/{vmid}/template
/// - /nodes/{node}/lxc/{vmid}/termproxy
/// - /nodes/{node}/lxc/{vmid}/vncproxy
/// - /nodes/{node}/lxc/{vmid}/vncwebsocket
/// - /nodes/{node}/migrateall
/// - /nodes/{node}/netstat
/// - /nodes/{node}/network/{iface}
/// - /nodes/{node}/qemu/{vmid}
/// - /nodes/{node}/qemu/{vmid}/agent
/// - /nodes/{node}/qemu/{vmid}/agent/exec
/// - /nodes/{node}/qemu/{vmid}/agent/exec-status
/// - /nodes/{node}/qemu/{vmid}/agent/file-read
/// - /nodes/{node}/qemu/{vmid}/agent/file-write
/// - /nodes/{node}/qemu/{vmid}/agent/fsfreeze-freeze
/// - /nodes/{node}/qemu/{vmid}/agent/fsfreeze-status
/// - /nodes/{node}/qemu/{vmid}/agent/fsfreeze-thaw
/// - /nodes/{node}/qemu/{vmid}/agent/fstrim
/// - /nodes/{node}/qemu/{vmid}/agent/get-fsinfo
/// - /nodes/{node}/qemu/{vmid}/agent/get-host-name
/// - /nodes/{node}/qemu/{vmid}/agent/get-memory-block-info
/// - /nodes/{node}/qemu/{vmid}/agent/get-memory-blocks
/// - /nodes/{node}/qemu/{vmid}/agent/get-osinfo
/// - /nodes/{node}/qemu/{vmid}/agent/get-time
/// - /nodes/{node}/qemu/{vmid}/agent/get-timezone
/// - /nodes/{node}/qemu/{vmid}/agent/get-users
/// - /nodes/{node}/qemu/{vmid}/agent/get-vcpus
/// - /nodes/{node}/qemu/{vmid}/agent/info
/// - /nodes/{node}/qemu/{vmid}/agent/network-get-interfaces
/// - /nodes/{node}/qemu/{vmid}/agent/ping
/// - /nodes/{node}/qemu/{vmid}/agent/set-user-password
/// - /nodes/{node}/qemu/{vmid}/agent/shutdown
/// - /nodes/{node}/qemu/{vmid}/agent/suspend-disk
/// - /nodes/{node}/qemu/{vmid}/agent/suspend-hybrid
/// - /nodes/{node}/qemu/{vmid}/agent/suspend-ram
/// - /nodes/{node}/qemu/{vmid}/clone
/// - /nodes/{node}/qemu/{vmid}/cloudinit
/// - /nodes/{node}/qemu/{vmid}/cloudinit/dump
/// - /nodes/{node}/qemu/{vmid}/dbus-vmstate
/// - /nodes/{node}/qemu/{vmid}/feature
/// - /nodes/{node}/qemu/{vmid}/firewall
/// - /nodes/{node}/qemu/{vmid}/firewall/rules/{pos}
/// - /nodes/{node}/qemu/{vmid}/monitor
/// - /nodes/{node}/qemu/{vmid}/mtunnel
/// - /nodes/{node}/qemu/{vmid}/mtunnelwebsocket
/// - /nodes/{node}/qemu/{vmid}/rrd
/// - /nodes/{node}/qemu/{vmid}/rrddata
/// - /nodes/{node}/qemu/{vmid}/sendkey
/// - /nodes/{node}/qemu/{vmid}/snapshot
/// - /nodes/{node}/qemu/{vmid}/snapshot/{snapname}
/// - /nodes/{node}/qemu/{vmid}/snapshot/{snapname}/config
/// - /nodes/{node}/qemu/{vmid}/snapshot/{snapname}/rollback
/// - /nodes/{node}/qemu/{vmid}/spiceproxy
/// - /nodes/{node}/qemu/{vmid}/status
/// - /nodes/{node}/qemu/{vmid}/status/reboot
/// - /nodes/{node}/qemu/{vmid}/status/reset
/// - /nodes/{node}/qemu/{vmid}/status/resume
/// - /nodes/{node}/qemu/{vmid}/status/suspend
/// - /nodes/{node}/qemu/{vmid}/template
/// - /nodes/{node}/qemu/{vmid}/termproxy
/// - /nodes/{node}/qemu/{vmid}/unlink
/// - /nodes/{node}/qemu/{vmid}/vncproxy
/// - /nodes/{node}/qemu/{vmid}/vncwebsocket
/// - /nodes/{node}/query-oci-repo-tags
/// - /nodes/{node}/query-url-metadata
/// - /nodes/{node}/replication
/// - /nodes/{node}/replication/{id}
/// - /nodes/{node}/replication/{id}/log
/// - /nodes/{node}/replication/{id}/schedule_now
/// - /nodes/{node}/replication/{id}/status
/// - /nodes/{node}/report
/// - /nodes/{node}/rrd
/// - /nodes/{node}/rrddata
/// - /nodes/{node}/scan
/// - /nodes/{node}/scan/cifs
/// - /nodes/{node}/scan/iscsi
/// - /nodes/{node}/scan/lvm
/// - /nodes/{node}/scan/lvmthin
/// - /nodes/{node}/scan/nfs
/// - /nodes/{node}/scan/pbs
/// - /nodes/{node}/scan/zfs
/// - /nodes/{node}/sdn
/// - /nodes/{node}/sdn/fabrics
/// - /nodes/{node}/sdn/fabrics/{fabric}
/// - /nodes/{node}/sdn/fabrics/{fabric}/interfaces
/// - /nodes/{node}/sdn/fabrics/{fabric}/neighbors
/// - /nodes/{node}/sdn/fabrics/{fabric}/routes
/// - /nodes/{node}/sdn/vnets
/// - /nodes/{node}/sdn/vnets/{vnet}
/// - /nodes/{node}/sdn/zones
/// - /nodes/{node}/sdn/zones/{zone}
/// - /nodes/{node}/sdn/zones/{zone}/bridges
/// - /nodes/{node}/sdn/zones/{zone}/content
/// - /nodes/{node}/services
/// - /nodes/{node}/services/{service}
/// - /nodes/{node}/services/{service}/reload
/// - /nodes/{node}/services/{service}/restart
/// - /nodes/{node}/services/{service}/start
/// - /nodes/{node}/services/{service}/state
/// - /nodes/{node}/services/{service}/stop
/// - /nodes/{node}/spiceshell
/// - /nodes/{node}/startall
/// - /nodes/{node}/stopall
/// - /nodes/{node}/storage/{storage}
/// - /nodes/{node}/storage/{storage}/content
/// - /nodes/{node}/storage/{storage}/content/{volume}
/// - /nodes/{node}/storage/{storage}/download-url
/// - /nodes/{node}/storage/{storage}/file-restore
/// - /nodes/{node}/storage/{storage}/file-restore/download
/// - /nodes/{node}/storage/{storage}/file-restore/list
/// - /nodes/{node}/storage/{storage}/import-metadata
/// - /nodes/{node}/storage/{storage}/oci-registry-pull
/// - /nodes/{node}/storage/{storage}/prunebackups
/// - /nodes/{node}/storage/{storage}/rrd
/// - /nodes/{node}/storage/{storage}/rrddata
/// - /nodes/{node}/storage/{storage}/upload
/// - /nodes/{node}/suspendall
/// - /nodes/{node}/syslog
/// - /nodes/{node}/time
/// - /nodes/{node}/version
/// - /nodes/{node}/vncshell
/// - /nodes/{node}/vncwebsocket
/// - /nodes/{node}/vzdump
/// - /nodes/{node}/vzdump/defaults
/// - /nodes/{node}/vzdump/extractconfig
/// - /nodes/{node}/wakeonlan
/// - /pools
/// - /pools/{poolid}
/// - /storage
/// - /storage/{storage}
/// ```
#[async_trait::async_trait]
pub trait PveClient {
    /// Acquire global lock for SDN configuration
    async fn acquire_sdn_lock(&self, params: CreateSdnLock) -> Result<String, Error> {
        Err(Error::Other("acquire_sdn_lock not implemented"))
    }

    /// Get information needed to join this cluster over the connected node.
    async fn cluster_config_join(&self, node: Option<String>) -> Result<ClusterJoinInfo, Error> {
        Err(Error::Other("cluster_config_join not implemented"))
    }

    /// List aliases
    async fn cluster_firewall_aliases(&self) -> Result<Vec<FirewallAlias>, Error> {
        Err(Error::Other("cluster_firewall_aliases not implemented"))
    }

    /// List IPSet content
    async fn cluster_firewall_ipset(&self, name: &str) -> Result<Vec<FirewallIpSet>, Error> {
        Err(Error::Other("cluster_firewall_ipset not implemented"))
    }

    /// Read IP or Network settings from IPSet.
    async fn cluster_firewall_ipset_entry(
        &self,
        name: &str,
        cidr: &str,
    ) -> Result<serde_json::Value, Error> {
        Err(Error::Other("cluster_firewall_ipset_entry not implemented"))
    }

    /// List IPSets
    async fn cluster_firewall_ipset_list(&self) -> Result<Vec<FirewallIpSetListItem>, Error> {
        Err(Error::Other("cluster_firewall_ipset_list not implemented"))
    }

    /// List available macros
    async fn cluster_firewall_macros(&self) -> Result<Vec<FirewallMacro>, Error> {
        Err(Error::Other("cluster_firewall_macros not implemented"))
    }

    /// Get Firewall options.
    async fn cluster_firewall_options(&self) -> Result<ClusterFirewallOptions, Error> {
        Err(Error::Other("cluster_firewall_options not implemented"))
    }

    /// Lists possible IPSet/Alias reference which are allowed in source/dest
    /// properties.
    async fn cluster_firewall_refs(
        &self,
        ty: Option<ClusterFirewallRefsType>,
    ) -> Result<Vec<FirewallRef>, Error> {
        Err(Error::Other("cluster_firewall_refs not implemented"))
    }

    /// Retrieve metrics of the cluster.
    async fn cluster_metrics_export(
        &self,
        history: Option<bool>,
        local_only: Option<bool>,
        node_list: Option<String>,
        start_time: Option<i64>,
    ) -> Result<ClusterMetrics, Error> {
        Err(Error::Other("cluster_metrics_export not implemented"))
    }

    /// Get datacenter options. Without 'Sys.Audit' on '/' not all options are
    /// returned.
    async fn cluster_options(&self) -> Result<serde_json::Value, Error> {
        Err(Error::Other("cluster_options not implemented"))
    }

    /// Resources index (cluster wide).
    async fn cluster_resources(
        &self,
        ty: Option<ClusterResourceKind>,
    ) -> Result<Vec<ClusterResource>, Error> {
        Err(Error::Other("cluster_resources not implemented"))
    }

    /// Get cluster status information.
    async fn cluster_status(&self) -> Result<Vec<ClusterNodeStatus>, Error> {
        Err(Error::Other("cluster_status not implemented"))
    }

    /// Create IP or Network Alias.
    async fn create_cluster_firewall_alias(
        &self,
        params: CreateFirewallAlias,
    ) -> Result<(), Error> {
        Err(Error::Other(
            "create_cluster_firewall_alias not implemented",
        ))
    }

    /// Add IP or Network to IPSet.
    async fn create_cluster_firewall_ipset_entry(
        &self,
        name: &str,
        params: CreateFirewallIpSetEntry,
    ) -> Result<(), Error> {
        Err(Error::Other(
            "create_cluster_firewall_ipset_entry not implemented",
        ))
    }

    /// Create a new sdn controller object.
    async fn create_controller(&self, params: CreateController) -> Result<(), Error> {
        Err(Error::Other("create_controller not implemented"))
    }

    /// Create IP or Network Alias.
    async fn create_lxc_firewall_alias(
        &self,
        node: &str,
        vmid: u32,
        params: CreateFirewallAlias,
    ) -> Result<(), Error> {
        Err(Error::Other("create_lxc_firewall_alias not implemented"))
    }

    /// Add IP or Network to IPSet.
    async fn create_lxc_firewall_ipset_entry(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        params: CreateFirewallIpSetEntry,
    ) -> Result<(), Error> {
        Err(Error::Other(
            "create_lxc_firewall_ipset_entry not implemented",
        ))
    }

    /// Create IP or Network Alias.
    async fn create_qemu_firewall_alias(
        &self,
        node: &str,
        vmid: u32,
        params: CreateFirewallAlias,
    ) -> Result<(), Error> {
        Err(Error::Other("create_qemu_firewall_alias not implemented"))
    }

    /// Add IP or Network to IPSet.
    async fn create_qemu_firewall_ipset_entry(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        params: CreateFirewallIpSetEntry,
    ) -> Result<(), Error> {
        Err(Error::Other(
            "create_qemu_firewall_ipset_entry not implemented",
        ))
    }

    /// Generate a new API token for a specific user. NOTE: returns API token
    /// value, which needs to be stored as it cannot be retrieved afterwards!
    async fn create_token(
        &self,
        userid: &str,
        tokenid: &str,
        params: CreateToken,
    ) -> Result<CreateTokenResponse, Error> {
        Err(Error::Other("create_token not implemented"))
    }

    /// Create a new sdn vnet object.
    async fn create_vnet(&self, params: CreateVnet) -> Result<(), Error> {
        Err(Error::Other("create_vnet not implemented"))
    }

    /// Create a new sdn zone object.
    async fn create_zone(&self, params: CreateZone) -> Result<(), Error> {
        Err(Error::Other("create_zone not implemented"))
    }

    /// Remove IP or Network alias.
    async fn delete_cluster_firewall_alias(
        &self,
        name: &str,
        params: DeleteFirewallAlias,
    ) -> Result<(), Error> {
        Err(Error::Other(
            "delete_cluster_firewall_alias not implemented",
        ))
    }

    /// Delete IPSet
    async fn delete_cluster_firewall_ipset(
        &self,
        name: &str,
        params: DeleteFirewallIpSet,
    ) -> Result<(), Error> {
        Err(Error::Other(
            "delete_cluster_firewall_ipset not implemented",
        ))
    }

    /// Remove IP or Network from IPSet.
    async fn delete_cluster_firewall_ipset_entry(
        &self,
        name: &str,
        cidr: &str,
        params: DeleteFirewallIpSetEntry,
    ) -> Result<(), Error> {
        Err(Error::Other(
            "delete_cluster_firewall_ipset_entry not implemented",
        ))
    }

    /// Remove IP or Network alias.
    async fn delete_lxc_firewall_alias(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        params: DeleteFirewallAlias,
    ) -> Result<(), Error> {
        Err(Error::Other("delete_lxc_firewall_alias not implemented"))
    }

    /// Delete IPSet
    async fn delete_lxc_firewall_ipset(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        params: DeleteFirewallIpSet,
    ) -> Result<(), Error> {
        Err(Error::Other("delete_lxc_firewall_ipset not implemented"))
    }

    /// Remove IP or Network from IPSet.
    async fn delete_lxc_firewall_ipset_entry(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        cidr: &str,
        params: DeleteFirewallIpSetEntry,
    ) -> Result<(), Error> {
        Err(Error::Other(
            "delete_lxc_firewall_ipset_entry not implemented",
        ))
    }

    /// Remove IP or Network alias.
    async fn delete_qemu_firewall_alias(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        params: DeleteFirewallAlias,
    ) -> Result<(), Error> {
        Err(Error::Other("delete_qemu_firewall_alias not implemented"))
    }

    /// Delete IPSet
    async fn delete_qemu_firewall_ipset(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        params: DeleteFirewallIpSet,
    ) -> Result<(), Error> {
        Err(Error::Other("delete_qemu_firewall_ipset not implemented"))
    }

    /// Remove IP or Network from IPSet.
    async fn delete_qemu_firewall_ipset_entry(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        cidr: &str,
        params: DeleteFirewallIpSetEntry,
    ) -> Result<(), Error> {
        Err(Error::Other(
            "delete_qemu_firewall_ipset_entry not implemented",
        ))
    }

    /// Get APT repository information.
    async fn get_apt_repositories(&self, node: &str) -> Result<APTRepositoriesResult, Error> {
        Err(Error::Other("get_apt_repositories not implemented"))
    }

    /// Get package changelogs.
    async fn get_package_changelog(
        &self,
        node: &str,
        name: String,
        version: Option<String>,
    ) -> Result<String, Error> {
        Err(Error::Other("get_package_changelog not implemented"))
    }

    /// Get package information for important Proxmox packages.
    async fn get_package_versions(&self, node: &str) -> Result<Vec<InstalledPackage>, Error> {
        Err(Error::Other("get_package_versions not implemented"))
    }

    /// Read subscription info.
    async fn get_subscription(&self, node: &str) -> Result<NodeSubscriptionInfo, Error> {
        Err(Error::Other("get_subscription not implemented"))
    }

    /// Read task list for one node (finished tasks).
    async fn get_task_list(
        &self,
        node: &str,
        params: ListTasks,
    ) -> Result<Vec<ListTasksResponse>, Error> {
        Err(Error::Other("get_task_list not implemented"))
    }

    /// Read task log.
    async fn get_task_log(
        &self,
        node: &str,
        upid: &str,
        download: Option<bool>,
        limit: Option<u64>,
        start: Option<u64>,
    ) -> Result<ApiResponseData<Vec<TaskLogLine>>, Error> {
        Err(Error::Other("get_task_log not implemented"))
    }

    /// Read task status.
    async fn get_task_status(&self, node: &str, upid: &str) -> Result<TaskStatus, Error> {
        Err(Error::Other("get_task_status not implemented"))
    }

    /// Get the MAC VRF for a VNet in an EVPN zone.
    async fn get_vnet_mac_vrf(&self, node: &str, vnet: &str) -> Result<Vec<SdnVnetMacVrf>, Error> {
        Err(Error::Other("get_vnet_mac_vrf not implemented"))
    }

    /// Get the IP VRF of an EVPN zone.
    async fn get_zone_ip_vrf(&self, node: &str, zone: &str) -> Result<Vec<SdnZoneIpVrf>, Error> {
        Err(Error::Other("get_zone_ip_vrf not implemented"))
    }

    /// List available updates.
    async fn list_available_updates(&self, node: &str) -> Result<Vec<AptUpdateInfo>, Error> {
        Err(Error::Other("list_available_updates not implemented"))
    }

    /// List rules.
    async fn list_cluster_firewall_rules(&self) -> Result<Vec<ListFirewallRules>, Error> {
        Err(Error::Other("list_cluster_firewall_rules not implemented"))
    }

    /// SDN controllers index.
    async fn list_controllers(
        &self,
        pending: Option<bool>,
        running: Option<bool>,
        ty: Option<ListControllersType>,
    ) -> Result<Vec<SdnController>, Error> {
        Err(Error::Other("list_controllers not implemented"))
    }

    /// Authentication domain index.
    async fn list_domains(&self) -> Result<Vec<ListRealm>, Error> {
        Err(Error::Other("list_domains not implemented"))
    }

    /// LXC container index (per node).
    async fn list_lxc(&self, node: &str) -> Result<Vec<LxcEntry>, Error> {
        Err(Error::Other("list_lxc not implemented"))
    }

    /// List rules.
    async fn list_lxc_firewall_rules(
        &self,
        node: &str,
        vmid: u32,
    ) -> Result<Vec<ListFirewallRules>, Error> {
        Err(Error::Other("list_lxc_firewall_rules not implemented"))
    }

    /// List available networks
    async fn list_networks(
        &self,
        node: &str,
        ty: Option<ListNetworksType>,
    ) -> Result<Vec<NetworkInterface>, Error> {
        Err(Error::Other("list_networks not implemented"))
    }

    /// List rules.
    async fn list_node_firewall_rules(&self, node: &str) -> Result<Vec<ListFirewallRules>, Error> {
        Err(Error::Other("list_node_firewall_rules not implemented"))
    }

    /// Cluster node index.
    async fn list_nodes(&self) -> Result<Vec<ClusterNodeIndexResponse>, Error> {
        Err(Error::Other("list_nodes not implemented"))
    }

    /// Virtual machine index (per node).
    async fn list_qemu(&self, node: &str, full: Option<bool>) -> Result<Vec<VmEntry>, Error> {
        Err(Error::Other("list_qemu not implemented"))
    }

    /// List rules.
    async fn list_qemu_firewall_rules(
        &self,
        node: &str,
        vmid: u32,
    ) -> Result<Vec<ListFirewallRules>, Error> {
        Err(Error::Other("list_qemu_firewall_rules not implemented"))
    }

    /// Get status for all datastores.
    async fn list_storages(
        &self,
        node: &str,
        content: Option<Vec<StorageContent>>,
        enabled: Option<bool>,
        format: Option<bool>,
        storage: Option<String>,
        target: Option<String>,
    ) -> Result<Vec<StorageInfo>, Error> {
        Err(Error::Other("list_storages not implemented"))
    }

    /// SDN vnets index.
    async fn list_vnets(
        &self,
        pending: Option<bool>,
        running: Option<bool>,
    ) -> Result<Vec<SdnVnet>, Error> {
        Err(Error::Other("list_vnets not implemented"))
    }

    /// SDN zones index.
    async fn list_zones(
        &self,
        pending: Option<bool>,
        running: Option<bool>,
        ty: Option<ListZonesType>,
    ) -> Result<Vec<SdnZone>, Error> {
        Err(Error::Other("list_zones not implemented"))
    }

    /// List aliases
    async fn lxc_firewall_aliases(
        &self,
        node: &str,
        vmid: u32,
    ) -> Result<Vec<FirewallAlias>, Error> {
        Err(Error::Other("lxc_firewall_aliases not implemented"))
    }

    /// List IPSet content
    async fn lxc_firewall_ipset(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
    ) -> Result<Vec<FirewallIpSet>, Error> {
        Err(Error::Other("lxc_firewall_ipset not implemented"))
    }

    /// Read IP or Network settings from IPSet.
    async fn lxc_firewall_ipset_entry(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        cidr: &str,
    ) -> Result<serde_json::Value, Error> {
        Err(Error::Other("lxc_firewall_ipset_entry not implemented"))
    }

    /// List IPSets
    async fn lxc_firewall_ipset_list(
        &self,
        node: &str,
        vmid: u32,
    ) -> Result<Vec<FirewallIpSetListItem>, Error> {
        Err(Error::Other("lxc_firewall_ipset_list not implemented"))
    }

    /// Read firewall log
    async fn lxc_firewall_log(
        &self,
        node: &str,
        vmid: u32,
        limit: Option<u64>,
        since: Option<u64>,
        start: Option<u64>,
        until: Option<u64>,
    ) -> Result<ApiResponseData<Vec<TaskLogLine>>, Error> {
        Err(Error::Other("lxc_firewall_log not implemented"))
    }

    /// Get VM firewall options.
    async fn lxc_firewall_options(
        &self,
        node: &str,
        vmid: u32,
    ) -> Result<GuestFirewallOptions, Error> {
        Err(Error::Other("lxc_firewall_options not implemented"))
    }

    /// Lists possible IPSet/Alias reference which are allowed in source/dest
    /// properties.
    async fn lxc_firewall_refs(
        &self,
        node: &str,
        vmid: u32,
        ty: Option<ClusterFirewallRefsType>,
    ) -> Result<Vec<FirewallRef>, Error> {
        Err(Error::Other("lxc_firewall_refs not implemented"))
    }

    /// Get container configuration.
    async fn lxc_get_config(
        &self,
        node: &str,
        vmid: u32,
        current: Option<bool>,
        snapshot: Option<String>,
    ) -> Result<LxcConfig, Error> {
        Err(Error::Other("lxc_get_config not implemented"))
    }

    /// Get container configuration, including pending changes.
    async fn lxc_get_pending(
        &self,
        node: &str,
        vmid: u32,
    ) -> Result<Vec<PendingConfigValue>, Error> {
        Err(Error::Other("lxc_get_pending not implemented"))
    }

    /// Get virtual machine status.
    async fn lxc_get_status(&self, node: &str, vmid: u32) -> Result<LxcStatus, Error> {
        Err(Error::Other("lxc_get_status not implemented"))
    }

    /// Move a rootfs-/mp-volume to a different storage or to a different
    /// container.
    async fn lxc_move_volume(
        &self,
        node: &str,
        vmid: u32,
        params: LxcMoveVolume,
    ) -> Result<PveUpid, Error> {
        Err(Error::Other("lxc_move_volume not implemented"))
    }

    /// Resize a container mount point.
    async fn lxc_resize(&self, node: &str, vmid: u32, params: LxcResize) -> Result<PveUpid, Error> {
        Err(Error::Other("lxc_resize not implemented"))
    }

    /// Set container options.
    async fn lxc_update_config(
        &self,
        node: &str,
        vmid: u32,
        params: UpdateLxcConfig,
    ) -> Result<(), Error> {
        Err(Error::Other("lxc_update_config not implemented"))
    }

    /// Migrate the container to another node. Creates a new migration task.
    async fn migrate_lxc(
        &self,
        node: &str,
        vmid: u32,
        params: MigrateLxc,
    ) -> Result<PveUpid, Error> {
        Err(Error::Other("migrate_lxc not implemented"))
    }

    /// Migrate virtual machine. Creates a new migration task.
    async fn migrate_qemu(
        &self,
        node: &str,
        vmid: u32,
        params: MigrateQemu,
    ) -> Result<PveUpid, Error> {
        Err(Error::Other("migrate_qemu not implemented"))
    }

    /// Get node configuration options.
    async fn node_config(
        &self,
        node: &str,
        property: Option<NodeConfigProperty>,
    ) -> Result<NodeConfig, Error> {
        Err(Error::Other("node_config not implemented"))
    }

    /// Read firewall log
    async fn node_firewall_log(
        &self,
        node: &str,
        limit: Option<u64>,
        since: Option<u64>,
        start: Option<u64>,
        until: Option<u64>,
    ) -> Result<ApiResponseData<Vec<TaskLogLine>>, Error> {
        Err(Error::Other("node_firewall_log not implemented"))
    }

    /// Get host firewall options.
    async fn node_firewall_options(&self, node: &str) -> Result<NodeFirewallOptions, Error> {
        Err(Error::Other("node_firewall_options not implemented"))
    }

    /// Creates a VNC Shell proxy.
    async fn node_shell_termproxy(
        &self,
        node: &str,
        params: NodeShellTermproxy,
    ) -> Result<NodeShellTicket, Error> {
        Err(Error::Other("node_shell_termproxy not implemented"))
    }

    /// Read node status
    async fn node_status(&self, node: &str) -> Result<NodeStatus, Error> {
        Err(Error::Other("node_status not implemented"))
    }

    /// List all custom and default CPU models.
    async fn qemu_cpu_capabilities(&self, node: &str) -> Result<Vec<QemuCpuModel>, Error> {
        Err(Error::Other("qemu_cpu_capabilities not implemented"))
    }

    /// List aliases
    async fn qemu_firewall_aliases(
        &self,
        node: &str,
        vmid: u32,
    ) -> Result<Vec<FirewallAlias>, Error> {
        Err(Error::Other("qemu_firewall_aliases not implemented"))
    }

    /// List IPSet content
    async fn qemu_firewall_ipset(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
    ) -> Result<Vec<FirewallIpSet>, Error> {
        Err(Error::Other("qemu_firewall_ipset not implemented"))
    }

    /// Read IP or Network settings from IPSet.
    async fn qemu_firewall_ipset_entry(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        cidr: &str,
    ) -> Result<serde_json::Value, Error> {
        Err(Error::Other("qemu_firewall_ipset_entry not implemented"))
    }

    /// List IPSets
    async fn qemu_firewall_ipset_list(
        &self,
        node: &str,
        vmid: u32,
    ) -> Result<Vec<FirewallIpSetListItem>, Error> {
        Err(Error::Other("qemu_firewall_ipset_list not implemented"))
    }

    /// Read firewall log
    async fn qemu_firewall_log(
        &self,
        node: &str,
        vmid: u32,
        limit: Option<u64>,
        since: Option<u64>,
        start: Option<u64>,
        until: Option<u64>,
    ) -> Result<ApiResponseData<Vec<TaskLogLine>>, Error> {
        Err(Error::Other("qemu_firewall_log not implemented"))
    }

    /// Get VM firewall options.
    async fn qemu_firewall_options(
        &self,
        node: &str,
        vmid: u32,
    ) -> Result<GuestFirewallOptions, Error> {
        Err(Error::Other("qemu_firewall_options not implemented"))
    }

    /// Lists possible IPSet/Alias reference which are allowed in source/dest
    /// properties.
    async fn qemu_firewall_refs(
        &self,
        node: &str,
        vmid: u32,
        ty: Option<ClusterFirewallRefsType>,
    ) -> Result<Vec<FirewallRef>, Error> {
        Err(Error::Other("qemu_firewall_refs not implemented"))
    }

    /// Get the virtual machine configuration with pending configuration changes
    /// applied. Set the 'current' parameter to get the current configuration
    /// instead.
    async fn qemu_get_config(
        &self,
        node: &str,
        vmid: u32,
        current: Option<bool>,
        snapshot: Option<String>,
    ) -> Result<QemuConfig, Error> {
        Err(Error::Other("qemu_get_config not implemented"))
    }

    /// Get the virtual machine configuration with both current and pending
    /// values.
    async fn qemu_get_pending(
        &self,
        node: &str,
        vmid: u32,
    ) -> Result<Vec<PendingConfigValue>, Error> {
        Err(Error::Other("qemu_get_pending not implemented"))
    }

    /// Get virtual machine status.
    async fn qemu_get_status(&self, node: &str, vmid: u32) -> Result<QemuStatus, Error> {
        Err(Error::Other("qemu_get_status not implemented"))
    }

    /// Get preconditions for migration.
    async fn qemu_migrate_preconditions(
        &self,
        node: &str,
        vmid: u32,
        target: Option<String>,
    ) -> Result<QemuMigratePreconditions, Error> {
        Err(Error::Other("qemu_migrate_preconditions not implemented"))
    }

    /// Move volume to different storage or to a different VM.
    async fn qemu_move_disk(
        &self,
        node: &str,
        vmid: u32,
        params: QemuMoveDisk,
    ) -> Result<PveUpid, Error> {
        Err(Error::Other("qemu_move_disk not implemented"))
    }

    /// Extend volume size.
    async fn qemu_resize(
        &self,
        node: &str,
        vmid: u32,
        params: QemuResize,
    ) -> Result<PveUpid, Error> {
        Err(Error::Other("qemu_resize not implemented"))
    }

    /// Set virtual machine options (synchronous API) - You should consider
    /// using the POST method instead for any actions involving hotplug or
    /// storage allocation.
    async fn qemu_update_config(
        &self,
        node: &str,
        vmid: u32,
        params: UpdateQemuConfig,
    ) -> Result<(), Error> {
        Err(Error::Other("qemu_update_config not implemented"))
    }

    /// Set virtual machine options (asynchronous API).
    async fn qemu_update_config_async(
        &self,
        node: &str,
        vmid: u32,
        params: UpdateQemuConfigAsync,
    ) -> Result<Option<PveUpid>, Error> {
        Err(Error::Other("qemu_update_config_async not implemented"))
    }

    /// Release global lock for SDN configuration
    async fn release_sdn_lock(&self, params: ReleaseSdnLock) -> Result<(), Error> {
        Err(Error::Other("release_sdn_lock not implemented"))
    }

    /// Migrate the container to another cluster. Creates a new migration task.
    /// EXPERIMENTAL feature!
    async fn remote_migrate_lxc(
        &self,
        node: &str,
        vmid: u32,
        params: RemoteMigrateLxc,
    ) -> Result<PveUpid, Error> {
        Err(Error::Other("remote_migrate_lxc not implemented"))
    }

    /// Migrate virtual machine to a remote cluster. Creates a new migration
    /// task. EXPERIMENTAL feature!
    async fn remote_migrate_qemu(
        &self,
        node: &str,
        vmid: u32,
        params: RemoteMigrateQemu,
    ) -> Result<PveUpid, Error> {
        Err(Error::Other("remote_migrate_qemu not implemented"))
    }

    /// Rollback pending changes to SDN configuration
    async fn rollback_sdn_changes(&self, params: RollbackSdn) -> Result<(), Error> {
        Err(Error::Other("rollback_sdn_changes not implemented"))
    }

    /// Apply sdn controller changes && reload.
    async fn sdn_apply(&self, params: ReloadSdn) -> Result<PveUpid, Error> {
        Err(Error::Other("sdn_apply not implemented"))
    }

    /// Set Firewall options.
    async fn set_cluster_firewall_options(
        &self,
        params: UpdateClusterFirewallOptions,
    ) -> Result<(), Error> {
        Err(Error::Other("set_cluster_firewall_options not implemented"))
    }

    /// Set Firewall options.
    async fn set_lxc_firewall_options(
        &self,
        node: &str,
        vmid: u32,
        params: UpdateGuestFirewallOptions,
    ) -> Result<(), Error> {
        Err(Error::Other("set_lxc_firewall_options not implemented"))
    }

    /// Set Firewall options.
    async fn set_node_firewall_options(
        &self,
        node: &str,
        params: UpdateNodeFirewallOptions,
    ) -> Result<(), Error> {
        Err(Error::Other("set_node_firewall_options not implemented"))
    }

    /// Set Firewall options.
    async fn set_qemu_firewall_options(
        &self,
        node: &str,
        vmid: u32,
        params: UpdateGuestFirewallOptions,
    ) -> Result<(), Error> {
        Err(Error::Other("set_qemu_firewall_options not implemented"))
    }

    /// Shutdown the container. This will trigger a clean shutdown of the
    /// container, see lxc-stop(1) for details.
    async fn shutdown_lxc_async(
        &self,
        node: &str,
        vmid: u32,
        params: ShutdownLxc,
    ) -> Result<PveUpid, Error> {
        Err(Error::Other("shutdown_lxc_async not implemented"))
    }

    /// Shutdown virtual machine. This is similar to pressing the power button
    /// on a physical machine. This will send an ACPI event for the guest OS,
    /// which should then proceed to a clean shutdown.
    async fn shutdown_qemu_async(
        &self,
        node: &str,
        vmid: u32,
        params: ShutdownQemu,
    ) -> Result<PveUpid, Error> {
        Err(Error::Other("shutdown_qemu_async not implemented"))
    }

    /// Start the container.
    async fn start_lxc_async(
        &self,
        node: &str,
        vmid: u32,
        params: StartLxc,
    ) -> Result<PveUpid, Error> {
        Err(Error::Other("start_lxc_async not implemented"))
    }

    /// Start virtual machine.
    async fn start_qemu_async(
        &self,
        node: &str,
        vmid: u32,
        params: StartQemu,
    ) -> Result<PveUpid, Error> {
        Err(Error::Other("start_qemu_async not implemented"))
    }

    /// Stop the container. This will abruptly stop all processes running in the
    /// container.
    async fn stop_lxc_async(
        &self,
        node: &str,
        vmid: u32,
        params: StopLxc,
    ) -> Result<PveUpid, Error> {
        Err(Error::Other("stop_lxc_async not implemented"))
    }

    /// Stop virtual machine. The qemu process will exit immediately. This is
    /// akin to pulling the power plug of a running computer and may damage the
    /// VM data.
    async fn stop_qemu_async(
        &self,
        node: &str,
        vmid: u32,
        params: StopQemu,
    ) -> Result<PveUpid, Error> {
        Err(Error::Other("stop_qemu_async not implemented"))
    }

    /// Stop a task.
    async fn stop_task(&self, node: &str, upid: &str) -> Result<(), Error> {
        Err(Error::Other("stop_task not implemented"))
    }

    /// Read storage status.
    async fn storage_status(&self, node: &str, storage: &str) -> Result<StorageStatus, Error> {
        Err(Error::Other("storage_status not implemented"))
    }

    /// This is used to resynchronize the package index files from their sources
    /// (apt-get update).
    async fn update_apt_database(
        &self,
        node: &str,
        params: AptUpdateParams,
    ) -> Result<PveUpid, Error> {
        Err(Error::Other("update_apt_database not implemented"))
    }

    /// Update IP or Network alias.
    async fn update_cluster_firewall_alias(
        &self,
        name: &str,
        params: UpdateFirewallAlias,
    ) -> Result<(), Error> {
        Err(Error::Other(
            "update_cluster_firewall_alias not implemented",
        ))
    }

    /// Update IP or Network settings
    async fn update_cluster_firewall_ipset_entry(
        &self,
        name: &str,
        cidr: &str,
        params: UpdateFirewallIpSetEntry,
    ) -> Result<(), Error> {
        Err(Error::Other(
            "update_cluster_firewall_ipset_entry not implemented",
        ))
    }

    /// Update IP or Network alias.
    async fn update_lxc_firewall_alias(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        params: UpdateFirewallAlias,
    ) -> Result<(), Error> {
        Err(Error::Other("update_lxc_firewall_alias not implemented"))
    }

    /// Update IP or Network settings
    async fn update_lxc_firewall_ipset_entry(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        cidr: &str,
        params: UpdateFirewallIpSetEntry,
    ) -> Result<(), Error> {
        Err(Error::Other(
            "update_lxc_firewall_ipset_entry not implemented",
        ))
    }

    /// Update IP or Network alias.
    async fn update_qemu_firewall_alias(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        params: UpdateFirewallAlias,
    ) -> Result<(), Error> {
        Err(Error::Other("update_qemu_firewall_alias not implemented"))
    }

    /// Update IP or Network settings
    async fn update_qemu_firewall_ipset_entry(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        cidr: &str,
        params: UpdateFirewallIpSetEntry,
    ) -> Result<(), Error> {
        Err(Error::Other(
            "update_qemu_firewall_ipset_entry not implemented",
        ))
    }

    /// API version details, including some parts of the global datacenter
    /// config.
    async fn version(&self) -> Result<VersionResponse, Error> {
        Err(Error::Other("version not implemented"))
    }
}

#[async_trait::async_trait]
impl<T> PveClient for PveClientImpl<T>
where
    T: HttpApiClient + Send + Sync,
    for<'a> <T as HttpApiClient>::ResponseFuture<'a>: Send,
{
    /// Acquire global lock for SDN configuration
    async fn acquire_sdn_lock(&self, params: CreateSdnLock) -> Result<String, Error> {
        let url = "/api2/extjs/cluster/sdn/lock";
        Ok(self.0.post(url, &params).await?.expect_json()?.data)
    }

    /// Get information needed to join this cluster over the connected node.
    async fn cluster_config_join(&self, node: Option<String>) -> Result<ClusterJoinInfo, Error> {
        let url = &ApiPathBuilder::new("/api2/extjs/cluster/config/join")
            .maybe_arg("node", &node)
            .build();
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// List aliases
    async fn cluster_firewall_aliases(&self) -> Result<Vec<FirewallAlias>, Error> {
        let url = "/api2/extjs/cluster/firewall/aliases";
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// List IPSet content
    async fn cluster_firewall_ipset(&self, name: &str) -> Result<Vec<FirewallIpSet>, Error> {
        let url = &format!(
            "/api2/extjs/cluster/firewall/ipset/{}",
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Read IP or Network settings from IPSet.
    async fn cluster_firewall_ipset_entry(
        &self,
        name: &str,
        cidr: &str,
    ) -> Result<serde_json::Value, Error> {
        let url = &format!(
            "/api2/extjs/cluster/firewall/ipset/{}/{}",
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            percent_encode(cidr.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// List IPSets
    async fn cluster_firewall_ipset_list(&self) -> Result<Vec<FirewallIpSetListItem>, Error> {
        let url = "/api2/extjs/cluster/firewall/ipset";
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// List available macros
    async fn cluster_firewall_macros(&self) -> Result<Vec<FirewallMacro>, Error> {
        let url = "/api2/extjs/cluster/firewall/macros";
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Get Firewall options.
    async fn cluster_firewall_options(&self) -> Result<ClusterFirewallOptions, Error> {
        let url = "/api2/extjs/cluster/firewall/options";
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Lists possible IPSet/Alias reference which are allowed in source/dest
    /// properties.
    async fn cluster_firewall_refs(
        &self,
        ty: Option<ClusterFirewallRefsType>,
    ) -> Result<Vec<FirewallRef>, Error> {
        let url = &ApiPathBuilder::new("/api2/extjs/cluster/firewall/refs")
            .maybe_arg("type", &ty)
            .build();
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Retrieve metrics of the cluster.
    async fn cluster_metrics_export(
        &self,
        history: Option<bool>,
        local_only: Option<bool>,
        node_list: Option<String>,
        start_time: Option<i64>,
    ) -> Result<ClusterMetrics, Error> {
        let url = &ApiPathBuilder::new("/api2/extjs/cluster/metrics/export")
            .maybe_bool_arg("history", history)
            .maybe_bool_arg("local-only", local_only)
            .maybe_arg("node-list", &node_list)
            .maybe_arg("start-time", &start_time)
            .build();
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Get datacenter options. Without 'Sys.Audit' on '/' not all options are
    /// returned.
    async fn cluster_options(&self) -> Result<serde_json::Value, Error> {
        let url = "/api2/extjs/cluster/options";
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Resources index (cluster wide).
    async fn cluster_resources(
        &self,
        ty: Option<ClusterResourceKind>,
    ) -> Result<Vec<ClusterResource>, Error> {
        let url = &ApiPathBuilder::new("/api2/extjs/cluster/resources")
            .maybe_arg("type", &ty)
            .build();
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Get cluster status information.
    async fn cluster_status(&self) -> Result<Vec<ClusterNodeStatus>, Error> {
        let url = "/api2/extjs/cluster/status";
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Create IP or Network Alias.
    async fn create_cluster_firewall_alias(
        &self,
        params: CreateFirewallAlias,
    ) -> Result<(), Error> {
        let url = "/api2/extjs/cluster/firewall/aliases";
        self.0.post(url, &params).await?.nodata()
    }

    /// Add IP or Network to IPSet.
    async fn create_cluster_firewall_ipset_entry(
        &self,
        name: &str,
        params: CreateFirewallIpSetEntry,
    ) -> Result<(), Error> {
        let url = &format!(
            "/api2/extjs/cluster/firewall/ipset/{}",
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        self.0.post(url, &params).await?.nodata()
    }

    /// Create a new sdn controller object.
    async fn create_controller(&self, params: CreateController) -> Result<(), Error> {
        let url = "/api2/extjs/cluster/sdn/controllers";
        self.0.post(url, &params).await?.nodata()
    }

    /// Create IP or Network Alias.
    async fn create_lxc_firewall_alias(
        &self,
        node: &str,
        vmid: u32,
        params: CreateFirewallAlias,
    ) -> Result<(), Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/firewall/aliases",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        self.0.post(url, &params).await?.nodata()
    }

    /// Add IP or Network to IPSet.
    async fn create_lxc_firewall_ipset_entry(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        params: CreateFirewallIpSetEntry,
    ) -> Result<(), Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/firewall/ipset/{}",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid,
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        self.0.post(url, &params).await?.nodata()
    }

    /// Create IP or Network Alias.
    async fn create_qemu_firewall_alias(
        &self,
        node: &str,
        vmid: u32,
        params: CreateFirewallAlias,
    ) -> Result<(), Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/firewall/aliases",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        self.0.post(url, &params).await?.nodata()
    }

    /// Add IP or Network to IPSet.
    async fn create_qemu_firewall_ipset_entry(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        params: CreateFirewallIpSetEntry,
    ) -> Result<(), Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/firewall/ipset/{}",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid,
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        self.0.post(url, &params).await?.nodata()
    }

    /// Generate a new API token for a specific user. NOTE: returns API token
    /// value, which needs to be stored as it cannot be retrieved afterwards!
    async fn create_token(
        &self,
        userid: &str,
        tokenid: &str,
        params: CreateToken,
    ) -> Result<CreateTokenResponse, Error> {
        let url = &format!(
            "/api2/extjs/access/users/{}/token/{}",
            percent_encode(userid.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            percent_encode(tokenid.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.post(url, &params).await?.expect_json()?.data)
    }

    /// Create a new sdn vnet object.
    async fn create_vnet(&self, params: CreateVnet) -> Result<(), Error> {
        let url = "/api2/extjs/cluster/sdn/vnets";
        self.0.post(url, &params).await?.nodata()
    }

    /// Create a new sdn zone object.
    async fn create_zone(&self, params: CreateZone) -> Result<(), Error> {
        let url = "/api2/extjs/cluster/sdn/zones";
        self.0.post(url, &params).await?.nodata()
    }

    /// Remove IP or Network alias.
    async fn delete_cluster_firewall_alias(
        &self,
        name: &str,
        params: DeleteFirewallAlias,
    ) -> Result<(), Error> {
        let DeleteFirewallAlias { digest: p_digest } = params;

        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/cluster/firewall/aliases/{}",
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        ))
        .maybe_arg("digest", &p_digest)
        .build();
        self.0.delete(url).await?.nodata()
    }

    /// Delete IPSet
    async fn delete_cluster_firewall_ipset(
        &self,
        name: &str,
        params: DeleteFirewallIpSet,
    ) -> Result<(), Error> {
        let DeleteFirewallIpSet { force: p_force } = params;

        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/cluster/firewall/ipset/{}",
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        ))
        .maybe_bool_arg("force", p_force)
        .build();
        self.0.delete(url).await?.nodata()
    }

    /// Remove IP or Network from IPSet.
    async fn delete_cluster_firewall_ipset_entry(
        &self,
        name: &str,
        cidr: &str,
        params: DeleteFirewallIpSetEntry,
    ) -> Result<(), Error> {
        let DeleteFirewallIpSetEntry { digest: p_digest } = params;

        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/cluster/firewall/ipset/{}/{}",
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            percent_encode(cidr.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        ))
        .maybe_arg("digest", &p_digest)
        .build();
        self.0.delete(url).await?.nodata()
    }

    /// Remove IP or Network alias.
    async fn delete_lxc_firewall_alias(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        params: DeleteFirewallAlias,
    ) -> Result<(), Error> {
        let DeleteFirewallAlias { digest: p_digest } = params;

        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/lxc/{}/firewall/aliases/{}",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid,
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        ))
        .maybe_arg("digest", &p_digest)
        .build();
        self.0.delete(url).await?.nodata()
    }

    /// Delete IPSet
    async fn delete_lxc_firewall_ipset(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        params: DeleteFirewallIpSet,
    ) -> Result<(), Error> {
        let DeleteFirewallIpSet { force: p_force } = params;

        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/lxc/{}/firewall/ipset/{}",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid,
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        ))
        .maybe_bool_arg("force", p_force)
        .build();
        self.0.delete(url).await?.nodata()
    }

    /// Remove IP or Network from IPSet.
    async fn delete_lxc_firewall_ipset_entry(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        cidr: &str,
        params: DeleteFirewallIpSetEntry,
    ) -> Result<(), Error> {
        let DeleteFirewallIpSetEntry { digest: p_digest } = params;

        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/lxc/{}/firewall/ipset/{}/{}",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid,
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            percent_encode(cidr.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        ))
        .maybe_arg("digest", &p_digest)
        .build();
        self.0.delete(url).await?.nodata()
    }

    /// Remove IP or Network alias.
    async fn delete_qemu_firewall_alias(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        params: DeleteFirewallAlias,
    ) -> Result<(), Error> {
        let DeleteFirewallAlias { digest: p_digest } = params;

        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/qemu/{}/firewall/aliases/{}",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid,
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        ))
        .maybe_arg("digest", &p_digest)
        .build();
        self.0.delete(url).await?.nodata()
    }

    /// Delete IPSet
    async fn delete_qemu_firewall_ipset(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        params: DeleteFirewallIpSet,
    ) -> Result<(), Error> {
        let DeleteFirewallIpSet { force: p_force } = params;

        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/qemu/{}/firewall/ipset/{}",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid,
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        ))
        .maybe_bool_arg("force", p_force)
        .build();
        self.0.delete(url).await?.nodata()
    }

    /// Remove IP or Network from IPSet.
    async fn delete_qemu_firewall_ipset_entry(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        cidr: &str,
        params: DeleteFirewallIpSetEntry,
    ) -> Result<(), Error> {
        let DeleteFirewallIpSetEntry { digest: p_digest } = params;

        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/qemu/{}/firewall/ipset/{}/{}",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid,
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            percent_encode(cidr.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        ))
        .maybe_arg("digest", &p_digest)
        .build();
        self.0.delete(url).await?.nodata()
    }

    /// Get APT repository information.
    async fn get_apt_repositories(&self, node: &str) -> Result<APTRepositoriesResult, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/apt/repositories",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Get package changelogs.
    async fn get_package_changelog(
        &self,
        node: &str,
        name: String,
        version: Option<String>,
    ) -> Result<String, Error> {
        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/apt/changelog",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        ))
        .arg("name", &name)
        .maybe_arg("version", &version)
        .build();
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Get package information for important Proxmox packages.
    async fn get_package_versions(&self, node: &str) -> Result<Vec<InstalledPackage>, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/apt/versions",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Read subscription info.
    async fn get_subscription(&self, node: &str) -> Result<NodeSubscriptionInfo, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/subscription",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Read task list for one node (finished tasks).
    async fn get_task_list(
        &self,
        node: &str,
        params: ListTasks,
    ) -> Result<Vec<ListTasksResponse>, Error> {
        let ListTasks {
            errors: p_errors,
            limit: p_limit,
            since: p_since,
            source: p_source,
            start: p_start,
            statusfilter: p_statusfilter,
            typefilter: p_typefilter,
            until: p_until,
            userfilter: p_userfilter,
            vmid: p_vmid,
        } = params;

        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/tasks",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        ))
        .maybe_bool_arg("errors", p_errors)
        .maybe_arg("limit", &p_limit)
        .maybe_arg("since", &p_since)
        .maybe_arg("source", &p_source)
        .maybe_arg("start", &p_start)
        .maybe_list_arg("statusfilter", &p_statusfilter)
        .maybe_arg("typefilter", &p_typefilter)
        .maybe_arg("until", &p_until)
        .maybe_arg("userfilter", &p_userfilter)
        .maybe_arg("vmid", &p_vmid)
        .build();
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Read task log.
    async fn get_task_log(
        &self,
        node: &str,
        upid: &str,
        download: Option<bool>,
        limit: Option<u64>,
        start: Option<u64>,
    ) -> Result<ApiResponseData<Vec<TaskLogLine>>, Error> {
        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/tasks/{}/log",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            percent_encode(upid.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        ))
        .maybe_bool_arg("download", download)
        .maybe_arg("limit", &limit)
        .maybe_arg("start", &start)
        .build();
        self.0.get(url).await?.expect_json()
    }

    /// Read task status.
    async fn get_task_status(&self, node: &str, upid: &str) -> Result<TaskStatus, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/tasks/{}/status",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            percent_encode(upid.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Get the MAC VRF for a VNet in an EVPN zone.
    async fn get_vnet_mac_vrf(&self, node: &str, vnet: &str) -> Result<Vec<SdnVnetMacVrf>, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/sdn/vnets/{}/mac-vrf",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            percent_encode(vnet.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Get the IP VRF of an EVPN zone.
    async fn get_zone_ip_vrf(&self, node: &str, zone: &str) -> Result<Vec<SdnZoneIpVrf>, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/sdn/zones/{}/ip-vrf",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            percent_encode(zone.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// List available updates.
    async fn list_available_updates(&self, node: &str) -> Result<Vec<AptUpdateInfo>, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/apt/update",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// List rules.
    async fn list_cluster_firewall_rules(&self) -> Result<Vec<ListFirewallRules>, Error> {
        let url = "/api2/extjs/cluster/firewall/rules";
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// SDN controllers index.
    async fn list_controllers(
        &self,
        pending: Option<bool>,
        running: Option<bool>,
        ty: Option<ListControllersType>,
    ) -> Result<Vec<SdnController>, Error> {
        let url = &ApiPathBuilder::new("/api2/extjs/cluster/sdn/controllers")
            .maybe_bool_arg("pending", pending)
            .maybe_bool_arg("running", running)
            .maybe_arg("type", &ty)
            .build();
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Authentication domain index.
    async fn list_domains(&self) -> Result<Vec<ListRealm>, Error> {
        let url = "/api2/extjs/access/domains";
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// LXC container index (per node).
    async fn list_lxc(&self, node: &str) -> Result<Vec<LxcEntry>, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// List rules.
    async fn list_lxc_firewall_rules(
        &self,
        node: &str,
        vmid: u32,
    ) -> Result<Vec<ListFirewallRules>, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/firewall/rules",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// List available networks
    async fn list_networks(
        &self,
        node: &str,
        ty: Option<ListNetworksType>,
    ) -> Result<Vec<NetworkInterface>, Error> {
        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/network",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        ))
        .maybe_arg("type", &ty)
        .build();
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// List rules.
    async fn list_node_firewall_rules(&self, node: &str) -> Result<Vec<ListFirewallRules>, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/firewall/rules",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Cluster node index.
    async fn list_nodes(&self) -> Result<Vec<ClusterNodeIndexResponse>, Error> {
        let url = "/api2/extjs/nodes";
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Virtual machine index (per node).
    async fn list_qemu(&self, node: &str, full: Option<bool>) -> Result<Vec<VmEntry>, Error> {
        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/qemu",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        ))
        .maybe_bool_arg("full", full)
        .build();
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// List rules.
    async fn list_qemu_firewall_rules(
        &self,
        node: &str,
        vmid: u32,
    ) -> Result<Vec<ListFirewallRules>, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/firewall/rules",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Get status for all datastores.
    async fn list_storages(
        &self,
        node: &str,
        content: Option<Vec<StorageContent>>,
        enabled: Option<bool>,
        format: Option<bool>,
        storage: Option<String>,
        target: Option<String>,
    ) -> Result<Vec<StorageInfo>, Error> {
        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/storage",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        ))
        .maybe_list_arg("content", &content)
        .maybe_bool_arg("enabled", enabled)
        .maybe_bool_arg("format", format)
        .maybe_arg("storage", &storage)
        .maybe_arg("target", &target)
        .build();
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// SDN vnets index.
    async fn list_vnets(
        &self,
        pending: Option<bool>,
        running: Option<bool>,
    ) -> Result<Vec<SdnVnet>, Error> {
        let url = &ApiPathBuilder::new("/api2/extjs/cluster/sdn/vnets")
            .maybe_bool_arg("pending", pending)
            .maybe_bool_arg("running", running)
            .build();
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// SDN zones index.
    async fn list_zones(
        &self,
        pending: Option<bool>,
        running: Option<bool>,
        ty: Option<ListZonesType>,
    ) -> Result<Vec<SdnZone>, Error> {
        let url = &ApiPathBuilder::new("/api2/extjs/cluster/sdn/zones")
            .maybe_bool_arg("pending", pending)
            .maybe_bool_arg("running", running)
            .maybe_arg("type", &ty)
            .build();
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// List aliases
    async fn lxc_firewall_aliases(
        &self,
        node: &str,
        vmid: u32,
    ) -> Result<Vec<FirewallAlias>, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/firewall/aliases",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// List IPSet content
    async fn lxc_firewall_ipset(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
    ) -> Result<Vec<FirewallIpSet>, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/firewall/ipset/{}",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid,
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Read IP or Network settings from IPSet.
    async fn lxc_firewall_ipset_entry(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        cidr: &str,
    ) -> Result<serde_json::Value, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/firewall/ipset/{}/{}",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid,
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            percent_encode(cidr.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// List IPSets
    async fn lxc_firewall_ipset_list(
        &self,
        node: &str,
        vmid: u32,
    ) -> Result<Vec<FirewallIpSetListItem>, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/firewall/ipset",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Read firewall log
    async fn lxc_firewall_log(
        &self,
        node: &str,
        vmid: u32,
        limit: Option<u64>,
        since: Option<u64>,
        start: Option<u64>,
        until: Option<u64>,
    ) -> Result<ApiResponseData<Vec<TaskLogLine>>, Error> {
        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/lxc/{}/firewall/log",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        ))
        .maybe_arg("limit", &limit)
        .maybe_arg("since", &since)
        .maybe_arg("start", &start)
        .maybe_arg("until", &until)
        .build();
        self.0.get(url).await?.expect_json()
    }

    /// Get VM firewall options.
    async fn lxc_firewall_options(
        &self,
        node: &str,
        vmid: u32,
    ) -> Result<GuestFirewallOptions, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/firewall/options",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Lists possible IPSet/Alias reference which are allowed in source/dest
    /// properties.
    async fn lxc_firewall_refs(
        &self,
        node: &str,
        vmid: u32,
        ty: Option<ClusterFirewallRefsType>,
    ) -> Result<Vec<FirewallRef>, Error> {
        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/lxc/{}/firewall/refs",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        ))
        .maybe_arg("type", &ty)
        .build();
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Get container configuration.
    async fn lxc_get_config(
        &self,
        node: &str,
        vmid: u32,
        current: Option<bool>,
        snapshot: Option<String>,
    ) -> Result<LxcConfig, Error> {
        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/lxc/{}/config",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        ))
        .maybe_bool_arg("current", current)
        .maybe_arg("snapshot", &snapshot)
        .build();
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Get container configuration, including pending changes.
    async fn lxc_get_pending(
        &self,
        node: &str,
        vmid: u32,
    ) -> Result<Vec<PendingConfigValue>, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/pending",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Get virtual machine status.
    async fn lxc_get_status(&self, node: &str, vmid: u32) -> Result<LxcStatus, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/status/current",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Move a rootfs-/mp-volume to a different storage or to a different
    /// container.
    async fn lxc_move_volume(
        &self,
        node: &str,
        vmid: u32,
        params: LxcMoveVolume,
    ) -> Result<PveUpid, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/move_volume",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.post(url, &params).await?.expect_json()?.data)
    }

    /// Resize a container mount point.
    async fn lxc_resize(&self, node: &str, vmid: u32, params: LxcResize) -> Result<PveUpid, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/resize",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.put(url, &params).await?.expect_json()?.data)
    }

    /// Set container options.
    async fn lxc_update_config(
        &self,
        node: &str,
        vmid: u32,
        params: UpdateLxcConfig,
    ) -> Result<(), Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/config",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        self.0.put(url, &params).await?.nodata()
    }

    /// Migrate the container to another node. Creates a new migration task.
    async fn migrate_lxc(
        &self,
        node: &str,
        vmid: u32,
        params: MigrateLxc,
    ) -> Result<PveUpid, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/migrate",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.post(url, &params).await?.expect_json()?.data)
    }

    /// Migrate virtual machine. Creates a new migration task.
    async fn migrate_qemu(
        &self,
        node: &str,
        vmid: u32,
        params: MigrateQemu,
    ) -> Result<PveUpid, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/migrate",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.post(url, &params).await?.expect_json()?.data)
    }

    /// Get node configuration options.
    async fn node_config(
        &self,
        node: &str,
        property: Option<NodeConfigProperty>,
    ) -> Result<NodeConfig, Error> {
        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/config",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        ))
        .maybe_arg("property", &property)
        .build();
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Read firewall log
    async fn node_firewall_log(
        &self,
        node: &str,
        limit: Option<u64>,
        since: Option<u64>,
        start: Option<u64>,
        until: Option<u64>,
    ) -> Result<ApiResponseData<Vec<TaskLogLine>>, Error> {
        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/firewall/log",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        ))
        .maybe_arg("limit", &limit)
        .maybe_arg("since", &since)
        .maybe_arg("start", &start)
        .maybe_arg("until", &until)
        .build();
        self.0.get(url).await?.expect_json()
    }

    /// Get host firewall options.
    async fn node_firewall_options(&self, node: &str) -> Result<NodeFirewallOptions, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/firewall/options",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Creates a VNC Shell proxy.
    async fn node_shell_termproxy(
        &self,
        node: &str,
        params: NodeShellTermproxy,
    ) -> Result<NodeShellTicket, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/termproxy",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.post(url, &params).await?.expect_json()?.data)
    }

    /// Read node status
    async fn node_status(&self, node: &str) -> Result<NodeStatus, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/status",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// List all custom and default CPU models.
    async fn qemu_cpu_capabilities(&self, node: &str) -> Result<Vec<QemuCpuModel>, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/capabilities/qemu/cpu",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// List aliases
    async fn qemu_firewall_aliases(
        &self,
        node: &str,
        vmid: u32,
    ) -> Result<Vec<FirewallAlias>, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/firewall/aliases",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// List IPSet content
    async fn qemu_firewall_ipset(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
    ) -> Result<Vec<FirewallIpSet>, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/firewall/ipset/{}",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid,
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Read IP or Network settings from IPSet.
    async fn qemu_firewall_ipset_entry(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        cidr: &str,
    ) -> Result<serde_json::Value, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/firewall/ipset/{}/{}",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid,
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            percent_encode(cidr.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// List IPSets
    async fn qemu_firewall_ipset_list(
        &self,
        node: &str,
        vmid: u32,
    ) -> Result<Vec<FirewallIpSetListItem>, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/firewall/ipset",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Read firewall log
    async fn qemu_firewall_log(
        &self,
        node: &str,
        vmid: u32,
        limit: Option<u64>,
        since: Option<u64>,
        start: Option<u64>,
        until: Option<u64>,
    ) -> Result<ApiResponseData<Vec<TaskLogLine>>, Error> {
        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/qemu/{}/firewall/log",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        ))
        .maybe_arg("limit", &limit)
        .maybe_arg("since", &since)
        .maybe_arg("start", &start)
        .maybe_arg("until", &until)
        .build();
        self.0.get(url).await?.expect_json()
    }

    /// Get VM firewall options.
    async fn qemu_firewall_options(
        &self,
        node: &str,
        vmid: u32,
    ) -> Result<GuestFirewallOptions, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/firewall/options",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Lists possible IPSet/Alias reference which are allowed in source/dest
    /// properties.
    async fn qemu_firewall_refs(
        &self,
        node: &str,
        vmid: u32,
        ty: Option<ClusterFirewallRefsType>,
    ) -> Result<Vec<FirewallRef>, Error> {
        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/qemu/{}/firewall/refs",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        ))
        .maybe_arg("type", &ty)
        .build();
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Get the virtual machine configuration with pending configuration changes
    /// applied. Set the 'current' parameter to get the current configuration
    /// instead.
    async fn qemu_get_config(
        &self,
        node: &str,
        vmid: u32,
        current: Option<bool>,
        snapshot: Option<String>,
    ) -> Result<QemuConfig, Error> {
        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/qemu/{}/config",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        ))
        .maybe_bool_arg("current", current)
        .maybe_arg("snapshot", &snapshot)
        .build();
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Get the virtual machine configuration with both current and pending
    /// values.
    async fn qemu_get_pending(
        &self,
        node: &str,
        vmid: u32,
    ) -> Result<Vec<PendingConfigValue>, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/pending",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Get virtual machine status.
    async fn qemu_get_status(&self, node: &str, vmid: u32) -> Result<QemuStatus, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/status/current",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Get preconditions for migration.
    async fn qemu_migrate_preconditions(
        &self,
        node: &str,
        vmid: u32,
        target: Option<String>,
    ) -> Result<QemuMigratePreconditions, Error> {
        let url = &ApiPathBuilder::new(format!(
            "/api2/extjs/nodes/{}/qemu/{}/migrate",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        ))
        .maybe_arg("target", &target)
        .build();
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// Move volume to different storage or to a different VM.
    async fn qemu_move_disk(
        &self,
        node: &str,
        vmid: u32,
        params: QemuMoveDisk,
    ) -> Result<PveUpid, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/move_disk",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.post(url, &params).await?.expect_json()?.data)
    }

    /// Extend volume size.
    async fn qemu_resize(
        &self,
        node: &str,
        vmid: u32,
        params: QemuResize,
    ) -> Result<PveUpid, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/resize",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.put(url, &params).await?.expect_json()?.data)
    }

    /// Set virtual machine options (synchronous API) - You should consider
    /// using the POST method instead for any actions involving hotplug or
    /// storage allocation.
    async fn qemu_update_config(
        &self,
        node: &str,
        vmid: u32,
        params: UpdateQemuConfig,
    ) -> Result<(), Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/config",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        self.0.put(url, &params).await?.nodata()
    }

    /// Set virtual machine options (asynchronous API).
    async fn qemu_update_config_async(
        &self,
        node: &str,
        vmid: u32,
        params: UpdateQemuConfigAsync,
    ) -> Result<Option<PveUpid>, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/config",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.post(url, &params).await?.expect_json()?.data)
    }

    /// Release global lock for SDN configuration
    async fn release_sdn_lock(&self, params: ReleaseSdnLock) -> Result<(), Error> {
        let ReleaseSdnLock {
            force: p_force,
            lock_token: p_lock_token,
        } = params;

        let url = &ApiPathBuilder::new("/api2/extjs/cluster/sdn/lock")
            .maybe_bool_arg("force", p_force)
            .maybe_arg("lock-token", &p_lock_token)
            .build();
        self.0.delete(url).await?.nodata()
    }

    /// Migrate the container to another cluster. Creates a new migration task.
    /// EXPERIMENTAL feature!
    async fn remote_migrate_lxc(
        &self,
        node: &str,
        vmid: u32,
        params: RemoteMigrateLxc,
    ) -> Result<PveUpid, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/remote_migrate",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.post(url, &params).await?.expect_json()?.data)
    }

    /// Migrate virtual machine to a remote cluster. Creates a new migration
    /// task. EXPERIMENTAL feature!
    async fn remote_migrate_qemu(
        &self,
        node: &str,
        vmid: u32,
        params: RemoteMigrateQemu,
    ) -> Result<PveUpid, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/remote_migrate",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.post(url, &params).await?.expect_json()?.data)
    }

    /// Rollback pending changes to SDN configuration
    async fn rollback_sdn_changes(&self, params: RollbackSdn) -> Result<(), Error> {
        let url = "/api2/extjs/cluster/sdn/rollback";
        self.0.post(url, &params).await?.nodata()
    }

    /// Apply sdn controller changes && reload.
    async fn sdn_apply(&self, params: ReloadSdn) -> Result<PveUpid, Error> {
        let url = "/api2/extjs/cluster/sdn";
        Ok(self.0.put(url, &params).await?.expect_json()?.data)
    }

    /// Set Firewall options.
    async fn set_cluster_firewall_options(
        &self,
        params: UpdateClusterFirewallOptions,
    ) -> Result<(), Error> {
        let url = "/api2/extjs/cluster/firewall/options";
        self.0.put(url, &params).await?.nodata()
    }

    /// Set Firewall options.
    async fn set_lxc_firewall_options(
        &self,
        node: &str,
        vmid: u32,
        params: UpdateGuestFirewallOptions,
    ) -> Result<(), Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/firewall/options",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        self.0.put(url, &params).await?.nodata()
    }

    /// Set Firewall options.
    async fn set_node_firewall_options(
        &self,
        node: &str,
        params: UpdateNodeFirewallOptions,
    ) -> Result<(), Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/firewall/options",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        self.0.put(url, &params).await?.nodata()
    }

    /// Set Firewall options.
    async fn set_qemu_firewall_options(
        &self,
        node: &str,
        vmid: u32,
        params: UpdateGuestFirewallOptions,
    ) -> Result<(), Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/firewall/options",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        self.0.put(url, &params).await?.nodata()
    }

    /// Shutdown the container. This will trigger a clean shutdown of the
    /// container, see lxc-stop(1) for details.
    async fn shutdown_lxc_async(
        &self,
        node: &str,
        vmid: u32,
        params: ShutdownLxc,
    ) -> Result<PveUpid, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/status/shutdown",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.post(url, &params).await?.expect_json()?.data)
    }

    /// Shutdown virtual machine. This is similar to pressing the power button
    /// on a physical machine. This will send an ACPI event for the guest OS,
    /// which should then proceed to a clean shutdown.
    async fn shutdown_qemu_async(
        &self,
        node: &str,
        vmid: u32,
        params: ShutdownQemu,
    ) -> Result<PveUpid, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/status/shutdown",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.post(url, &params).await?.expect_json()?.data)
    }

    /// Start the container.
    async fn start_lxc_async(
        &self,
        node: &str,
        vmid: u32,
        params: StartLxc,
    ) -> Result<PveUpid, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/status/start",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.post(url, &params).await?.expect_json()?.data)
    }

    /// Start virtual machine.
    async fn start_qemu_async(
        &self,
        node: &str,
        vmid: u32,
        params: StartQemu,
    ) -> Result<PveUpid, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/status/start",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.post(url, &params).await?.expect_json()?.data)
    }

    /// Stop the container. This will abruptly stop all processes running in the
    /// container.
    async fn stop_lxc_async(
        &self,
        node: &str,
        vmid: u32,
        params: StopLxc,
    ) -> Result<PveUpid, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/status/stop",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.post(url, &params).await?.expect_json()?.data)
    }

    /// Stop virtual machine. The qemu process will exit immediately. This is
    /// akin to pulling the power plug of a running computer and may damage the
    /// VM data.
    async fn stop_qemu_async(
        &self,
        node: &str,
        vmid: u32,
        params: StopQemu,
    ) -> Result<PveUpid, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/status/stop",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid
        );
        Ok(self.0.post(url, &params).await?.expect_json()?.data)
    }

    /// Stop a task.
    async fn stop_task(&self, node: &str, upid: &str) -> Result<(), Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/tasks/{}",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            percent_encode(upid.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        self.0.delete(url).await?.nodata()
    }

    /// Read storage status.
    async fn storage_status(&self, node: &str, storage: &str) -> Result<StorageStatus, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/storage/{}/status",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            percent_encode(storage.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.get(url).await?.expect_json()?.data)
    }

    /// This is used to resynchronize the package index files from their sources
    /// (apt-get update).
    async fn update_apt_database(
        &self,
        node: &str,
        params: AptUpdateParams,
    ) -> Result<PveUpid, Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/apt/update",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        Ok(self.0.post(url, &params).await?.expect_json()?.data)
    }

    /// Update IP or Network alias.
    async fn update_cluster_firewall_alias(
        &self,
        name: &str,
        params: UpdateFirewallAlias,
    ) -> Result<(), Error> {
        let url = &format!(
            "/api2/extjs/cluster/firewall/aliases/{}",
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        self.0.put(url, &params).await?.nodata()
    }

    /// Update IP or Network settings
    async fn update_cluster_firewall_ipset_entry(
        &self,
        name: &str,
        cidr: &str,
        params: UpdateFirewallIpSetEntry,
    ) -> Result<(), Error> {
        let url = &format!(
            "/api2/extjs/cluster/firewall/ipset/{}/{}",
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            percent_encode(cidr.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        self.0.put(url, &params).await?.nodata()
    }

    /// Update IP or Network alias.
    async fn update_lxc_firewall_alias(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        params: UpdateFirewallAlias,
    ) -> Result<(), Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/firewall/aliases/{}",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid,
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        self.0.put(url, &params).await?.nodata()
    }

    /// Update IP or Network settings
    async fn update_lxc_firewall_ipset_entry(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        cidr: &str,
        params: UpdateFirewallIpSetEntry,
    ) -> Result<(), Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/lxc/{}/firewall/ipset/{}/{}",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid,
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            percent_encode(cidr.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        self.0.put(url, &params).await?.nodata()
    }

    /// Update IP or Network alias.
    async fn update_qemu_firewall_alias(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        params: UpdateFirewallAlias,
    ) -> Result<(), Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/firewall/aliases/{}",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid,
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        self.0.put(url, &params).await?.nodata()
    }

    /// Update IP or Network settings
    async fn update_qemu_firewall_ipset_entry(
        &self,
        node: &str,
        vmid: u32,
        name: &str,
        cidr: &str,
        params: UpdateFirewallIpSetEntry,
    ) -> Result<(), Error> {
        let url = &format!(
            "/api2/extjs/nodes/{}/qemu/{}/firewall/ipset/{}/{}",
            percent_encode(node.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            vmid,
            percent_encode(name.as_bytes(), percent_encoding::NON_ALPHANUMERIC),
            percent_encode(cidr.as_bytes(), percent_encoding::NON_ALPHANUMERIC)
        );
        self.0.put(url, &params).await?.nodata()
    }

    /// API version details, including some parts of the global datacenter
    /// config.
    async fn version(&self) -> Result<VersionResponse, Error> {
        let url = "/api2/extjs/version";
        Ok(self.0.get(url).await?.expect_json()?.data)
    }
}
