#!/usr/bin/env perl

use v5.36;

use Carp;
use IPC::Open2;

use Data::Dumper;
$Data::Dumper::Indent = 1;

use lib './generator-lib';
use Schema2Rust;
use ApiDump;

my $output_dir = shift(@ARGV) // die "usage: $0 <output-directory>\n";

sub sq : prototype($) {
    return Schema2Rust::quote_string($_[0]);
}

# Load API dump:
my $pve_api = ApiDump::load_api('pve-api.json');

my sub lookup_format :prototype($) ($format) {
    return $pve_api->{formats}->{$format} // die "missing format: '$format'\n";
}

# Initialize:
Schema2Rust::init_api($pve_api->{root}, \&lookup_format);

# From JSONSchema.pm, but we can't use perl-re directly, particularly `qr//`...
my $CONFIGID_RE = '^(?i:[a-z][a-z0-9_-]+)$';

my $STORAGEID_RE = '(?i:[a-z][a-z0-9\-_.]*[a-z0-9])';
my $BRIDGEID_RE = '[-_.\w\d]+';

# Disable `#[api]` generation for now, it's incomplete/untested.
#$Schema2Rust::API = 0;

Schema2Rust::register_format('CIDR' => { code => 'verifiers::verify_cidr' });
Schema2Rust::register_format('CIDRv4' => { code => 'verifiers::verify_cidrv4' });
Schema2Rust::register_format('CIDRv6' => { code => 'verifiers::verify_cidrv6' });
Schema2Rust::register_format('ipv4mask' => { code => 'verifiers::verify_ipv4_mask' });
Schema2Rust::register_format('IPorCIDR' => { code => 'verifiers::verify_ip_or_cidr' });
Schema2Rust::register_format('IPorCIDRorAlias' => { code => 'verifiers::verify_ip_or_cidr_or_alias' });

Schema2Rust::register_format('mac-addr' => { regex => '^(?i)[a-f0-9][02468ace](?::[a-f0-9]{2}){5}$' });
Schema2Rust::register_format('pve-acme-alias' => { code => 'verifiers::verify_pve_acme_alias' });
Schema2Rust::register_format('pve-acme-domain' => { code => 'verifiers::verify_pve_acme_domain' });
Schema2Rust::register_format('pve-bridge-id' => { regex => '^'.$BRIDGEID_RE.'$' });
Schema2Rust::register_format('pve-configid' => { regex => $CONFIGID_RE });
## Schema2Rust::register_format('pve-groupid' => { code => 'verify_pve_groupid' });
Schema2Rust::register_format('pve-userid' => { code => 'verify_pve_userid' });
## # copied from JSONSchema's verify_pve_node sub:
Schema2Rust::register_format('pve-node' => { regex => '^(?i:[a-z0-9](?i:[a-z0-9\-]*[a-z0-9])?)$' });
## #Schema2Rust::register_format('pve-node' => { code => 'verify_pve_node' });
## Schema2Rust::register_format('pve-priv' => { code => 'verify_pve_privileges' });
## Schema2Rust::register_format('pve-realm' => { code => 'verify_pve_realm' });
##
Schema2Rust::register_format('disk-size' => { regex => '^(\d+(\.\d+)?)([KMGT])?$' });
Schema2Rust::register_format('dns-name' => { code => 'verifiers::verify_dns_name' });
## Schema2Rust::register_format('email' => { code => 'verify_email' });
Schema2Rust::register_format('pve-phys-bits' => { code => 'verifiers::verify_pve_phys_bits' });
Schema2Rust::register_format('pve-qm-bootdev' => { unchecked => 1 });
Schema2Rust::register_format('pve-qm-bootdisk' => { regex => '^(ide|sata|scsi|virtio|efidisk|tpmstate)\d+$' });
Schema2Rust::register_format('pve-qm-usb-device' => { unchecked => 1 });
Schema2Rust::register_format('pve-startup-order' => { unchecked => 1 });
Schema2Rust::register_format('pve-storage-id' => { regex => '^'.$STORAGEID_RE.'$' });
Schema2Rust::register_format('pve-storage-content' => { type => 'StorageContent' });
Schema2Rust::register_format('pve-tag' => { regex => '^(?i)[a-z0-9_][a-z0-9_\-+.]*$' });
Schema2Rust::register_format('pve-volume-id' => { code => 'verifiers::verify_volume_id' });
Schema2Rust::register_format('pve-volume-id-or-qm-path' => { code => 'verifiers::verify_pve_volume_id_or_qm_path' });
Schema2Rust::register_format('pve-volume-id-or-absolute-path' => { code => 'verifiers::verify_pve_volume_id_or_absolute_path' });
## Schema2Rust::register_format('pve-volume-id-or-absolute-path' => { code => 'verify_pve_volume_id_or_absolute_path' });
Schema2Rust::register_format('urlencoded' => { regex => '^[-%a-zA-Z0-9_.!~*\'()]*$' });
Schema2Rust::register_format('pve-cpuset' => { regex => '^(\s*\d+(-\d+)?\s*)(,\s*\d+(-\d+)?\s*)?$' });
##
Schema2Rust::register_format('pve-lxc-mp-string' => { code => 'verifiers::verify_lxc_mp_string' });
## Schema2Rust::register_format('lxc-ip-with-ll-iface' => { regex => ['^(?i:', \'pdm_api_types::IPRE!()', ')$'] });
Schema2Rust::register_format('lxc-ip-with-ll-iface' => { code => 'verifiers::verify_ip_with_ll_iface' });
Schema2Rust::register_format('pve-ct-timezone' => { regex => '^.*/.*$' });
Schema2Rust::register_format('pve-lxc-dev-string' => { code => 'verifiers::verify_pve_lxc_dev_string' });
##
Schema2Rust::register_format('storage-pair' => { regex => '^'.$STORAGEID_RE.':'.$STORAGEID_RE.'|'.$STORAGEID_RE.'|1$' });
Schema2Rust::register_format('bridge-pair' => { regex => '^'.$BRIDGEID_RE.':'.$BRIDGEID_RE.'|'.$BRIDGEID_RE.'|1$' });

Schema2Rust::register_format('pve-task-status-type' => { regex => '^(?i:ok|error|warning|unknown)$' });

Schema2Rust::register_format('pve-sdn-zone-id' => { code => 'verifiers::verify_sdn_id' });
Schema2Rust::register_format('pve-sdn-vnet-id' => { code => 'verifiers::verify_sdn_id' });

Schema2Rust::register_enum_variant('PveVmCpuConfReportedModel::486' => 'I486');
Schema2Rust::register_enum_variant('QemuConfigEfidisk0Efitype::2m' => 'Mb2');
Schema2Rust::register_enum_variant('QemuConfigEfidisk0Efitype::4m' => 'Mb4');
Schema2Rust::register_enum_variant('QemuConfigEfidisk0MsCert::2011' => 'CA2011');
Schema2Rust::register_enum_variant('QemuConfigEfidisk0MsCert::2023' => 'CA2023');
Schema2Rust::register_enum_variant('QemuConfigHugepages::2' => 'Mb2');
Schema2Rust::register_enum_variant('QemuConfigHugepages::1024' => 'Mb1024');
Schema2Rust::register_enum_variant('PveQmRngSource::/dev/urandom', => 'DevUrandom');
Schema2Rust::register_enum_variant('PveQmRngSource::/dev/random', => 'DevRandom');
Schema2Rust::register_enum_variant('PveQmRngSource::/dev/hwrng', => 'DevHwrng');
Schema2Rust::register_enum_variant('QemuConfigTpmstate0Version::v1.2' => 'V1_2');
Schema2Rust::register_enum_variant('QemuConfigTpmstate0Version::v2.0' => 'V2_0');

## # FIXME: Invent an enum list type for this one
Schema2Rust::register_format('pve-hotplug-features' => { unchecked => 1 });
## # FIXME: Figure out something sane for these
Schema2Rust::register_format('address' => { code => 'verifiers::verify_address' });
Schema2Rust::register_format('ip' => { code => 'verifiers::verify_ip' });
Schema2Rust::register_format('ipv4' => { code => 'verifiers::verify_ipv4' });
Schema2Rust::register_format('ipv6' => { code => 'verifiers::verify_ipv6' });
Schema2Rust::register_format('pve-ipv4-config' => { code => 'verifiers::verify_ipv4_config' });
Schema2Rust::register_format('pve-ipv6-config' => { code => 'verifiers::verify_ipv6_config' });

Schema2Rust::register_format('pve-iface' => { regex => '^[a-zA-Z][a-zA-Z0-9_]{1,20}([:\.]\d+)?$' });

Schema2Rust::register_format('pve-vlan-id-or-range' => { code => 'verifiers::verify_vlan_id_or_range' });

Schema2Rust::register_format('pve-sdn-bgp-rt' => { code => 'verifiers::verify_sdn_bgp_rt' });
Schema2Rust::register_format('pve-sdn-controller-id' => { code => 'verifiers::verify_sdn_controller_id' });
Schema2Rust::register_format('pve-sdn-isis-net' => { regex => '^[a-fA-F0-9]{2}(\.[a-fA-F0-9]{4}){3,9}\.[a-fA-F0-9]{2}$' });
Schema2Rust::register_format('pve-sdn-fabric-id' => { code => 'verifiers::verify_sdn_id' });

# This is used as both a task status and guest status.
Schema2Rust::generate_enum('IsRunning', {
    type => 'string',
    description => "A guest's run state.",
    enum => ['running', 'stopped'],
});

# We have a textual description of the default value in there, just pick the cgroupv2 one:
Schema2Rust::register_api_override('UpdateQemuConfig', '/properties/cpuunits/default', 1024);
Schema2Rust::register_api_override('UpdateQemuConfigAsync', '/properties/cpuunits/default', 1024);
Schema2Rust::register_api_override('UpdateLxcConfig', '/properties/cpuunits/default', 100);

Schema2Rust::register_api_override('QemuConfig', '/properties/cpuunits/default', 1024);
Schema2Rust::register_api_override('LxcConfig', '/properties/cpuunits/default', 1024);
Schema2Rust::register_api_extension('LxcConfig', '/properties/lxc/items', {
    description => sq('A raw lxc config entry'),
});
Schema2Rust::register_api_extension('LxcConfig', '/properties/lxc/items/items', {
    description => sq('A config key value pair'),
});
Schema2Rust::register_api_override('StartQemu', '/properties/timeout/default', 30);
Schema2Rust::register_api_override('RemoteMigrateQemu', '/properties/bwlimit/default', undef);
Schema2Rust::register_api_override('RemoteMigrateLxc', '/properties/bwlimit/default', undef);

# We have a textual description of the default value in there, simply set to undef
Schema2Rust::register_api_override('QemuMoveDisk', '/properties/bwlimit/default', undef);
Schema2Rust::register_api_override('LxcMoveVolume', '/properties/bwlimit/default', undef);

# Token api is missing some descriptions and has textual defaults for integers
Schema2Rust::register_api_extensions('CreateTokenResponseInfo', {
    '/properties/comment' => { description => sq("Description of the Token") },
});
Schema2Rust::register_api_extensions('CreateToken', {
    '/properties/comment' => { description => sq("Description of the Token") },
});
Schema2Rust::register_api_override('CreateTokenResponseInfo', '/properties/expire/default', undef);
Schema2Rust::register_api_override('CreateToken', '/properties/expire/default', undef);

# The task API is missing most documentation...
Schema2Rust::register_api_extensions('TaskStatus', {
    '/properties/exitstatus' => { description => sq("The task's exit status.") },
    '/properties/id' => { description => sq("The task id.") },
    '/properties/node' => { description => sq("The task's node.") },
    '/properties/type' => { description => sq("The task type.") },
    '/properties/upid' => { description => sq("The task's UPID.") },
    '/properties/user' => { description => sq("The task owner's user id.") },
    '/properties/pid' => { description => sq("The task process id.") },
    '/properties/pstart' => { description => sq("The task's proc start time.") },
    '/properties/starttime' => { description => sq("The task's start time.") },
});
Schema2Rust::register_api_extensions('ListTasksResponse', {
    '/properties/endtime' => { description => sq("The task's end time.") },
    '/properties/id' => { description => sq("The task id.") },
    '/properties/node' => { description => sq("The task's node.") },
    '/properties/pid' => { description => sq("The task process id.") },
    '/properties/pstart' => { description => sq("The task's proc start time.") },
    '/properties/starttime' => { description => sq("The task's start time.") },
    '/properties/status' => { description => sq("The task's status.") },
    '/properties/type' => { description => sq("The task type.") },
    '/properties/upid' => { description => sq("The task's UPID.") },
    '/properties/user' => { description => sq("The task owner's user id.") },
});
Schema2Rust::register_api_extensions('ClusterResource', {
    '/properties/id' => { description => sq("Resource id.") },
});

# pve-storage-content uses verify()
Schema2Rust::generate_enum('StorageContent', {
    type => 'string',
    description => 'Storage content type.',
    enum => $pve_api->{'storage-content-types'},
});

sub api : prototype($$$;%) {
    my ($method, $api_url, $rust_method_name, %extra) = @_;
    return Schema2Rust::api($method, $api_url, $rust_method_name, %extra);
}

# FIXME: this needs the return schema specified first:
api(GET => '/version', 'version', 'return-name' => 'VersionResponse');

# Deal with 'type' in `/cluster/resources` being different between input and output.
Schema2Rust::generate_enum(
    'ClusterResourceKind',
    {
        type => 'string',
        description => 'Resource type.',
        enum => ['vm', 'storage', 'node', 'sdn'],
    }
);
api(GET => '/cluster/resources', 'cluster_resources', 'return-name' => 'ClusterResource');
Schema2Rust::derive('ClusterResource' => 'Clone', 'PartialEq');

api(GET => '/nodes', 'list_nodes', 'return-name' => 'ClusterNodeIndexResponse');
Schema2Rust::derive('ClusterNodeIndexResponse' => 'Clone', 'PartialEq');
api(GET => '/nodes/{node}/config', 'node_config', 'return-name' => 'NodeConfig');
# api(PUT => '/nodes/{node}/config', 'set_node_config', 'param-name' => 'UpdateNodeConfig');
# subscription api
#
Schema2Rust::register_enum_variant('NodeSubscriptionInfoStatus::notfound' => 'NotFound');
api(GET => '/nodes/{node}/subscription', 'get_subscription',   'return-name' => 'NodeSubscriptionInfo');
# # low level task api:
# # ?? api(GET    => '/nodes/{node}/tasks/{upid}', 'get_task');
api(GET => '/nodes/{node}/tasks',               'get_task_list',   'param-name' => 'ListTasks');
Schema2Rust::derive('ListTasks' => 'Default');
api(GET => '/nodes/{node}/tasks/{upid}/status', 'get_task_status', 'return-name' => 'TaskStatus');
api(GET => '/nodes/{node}/tasks/{upid}/log',    'get_task_log',    'return-name' => 'TaskLogLine', attribs => 1);
api(DELETE => '/nodes/{node}/tasks/{upid}',     'stop_task');

api(GET => '/nodes/{node}/qemu', 'list_qemu', 'param-name' => 'FixmeListQemu', 'return-name' => 'VmEntry');
api(GET => '/nodes/{node}/qemu/{vmid}/config', 'qemu_get_config', 'param-name' => 'FixmeQemuGetConfig', 'return-name' => 'QemuConfig');
api(GET => '/nodes/{node}/qemu/{vmid}/pending', 'qemu_get_pending', 'param-name' => 'FixmeQemuGetPending', 'output-type' => 'Vec<PendingConfigValue>');
api(GET => '/nodes/{node}/qemu/{vmid}/status/current', 'qemu_get_status', 'return-name' => 'QemuStatus');
api(PUT => '/nodes/{node}/qemu/{vmid}/config', 'qemu_update_config', 'param-name' => 'UpdateQemuConfig');
api(POST => '/nodes/{node}/qemu/{vmid}/config', 'qemu_update_config_async', 'param-name' => 'UpdateQemuConfigAsync', 'output-type' => 'Option<PveUpid>');
api(POST => '/nodes/{node}/qemu/{vmid}/move_disk', 'qemu_move_disk', 'param-name' => 'QemuMoveDisk', 'output-type' => 'PveUpid');
api(PUT => '/nodes/{node}/qemu/{vmid}/resize', 'qemu_resize', 'param-name' => 'QemuResize', 'output-type' => 'PveUpid');
api(POST => '/nodes/{node}/qemu/{vmid}/status/start',    'start_qemu_async',    'output-type' => 'PveUpid', 'param-name' => 'StartQemu');
api(POST => '/nodes/{node}/qemu/{vmid}/status/stop',     'stop_qemu_async',     'output-type' => 'PveUpid', 'param-name' => 'StopQemu');
api(POST => '/nodes/{node}/qemu/{vmid}/status/shutdown', 'shutdown_qemu_async', 'output-type' => 'PveUpid', 'param-name' => 'ShutdownQemu');

# PVE9 introduced a non-optional return value, mark it optional manually so we can still use it with PVE8
my $r = (Schema2Rust::get_return_type(GET => '/nodes/{node}/qemu/{vmid}/migrate'))->{properties};
$r->{'has-dbus-vmstate'}->{optional} = '1';
api(GET => '/nodes/{node}/qemu/{vmid}/migrate',           'qemu_migrate_preconditions', 'return-name' => 'QemuMigratePreconditions');
Schema2Rust::derive('QemuMigratePreconditionsNotAllowedNodesBlockingHaResources' => 'Clone', 'PartialEq');
Schema2Rust::derive('QemuMigratePreconditionsNotAllowedNodes' => 'Clone', 'PartialEq');
Schema2Rust::derive('QemuMigratePreconditionsLocalDisks' => 'Clone', 'PartialEq');
Schema2Rust::derive('QemuMigratePreconditions' => 'Clone', 'PartialEq');

Schema2Rust::derive('StartQemu' => 'Default');
Schema2Rust::derive('StopQemu' => 'Default');
Schema2Rust::derive('ShutdownQemu' => 'Default');
api(POST => '/nodes/{node}/qemu/{vmid}/migrate',        'migrate_qemu',         'output-type' => 'PveUpid', 'param-name' => 'MigrateQemu');
Schema2Rust::register_api_override('MigrateQemu', '/properties/bwlimit/default', undef);
api(POST => '/nodes/{node}/qemu/{vmid}/remote_migrate', 'remote_migrate_qemu',  'output-type' => 'PveUpid', 'param-name' => 'RemoteMigrateQemu');

api(GET => '/nodes/{node}/lxc',                         'list_lxc',            'param-name' => 'FixmeListLxc',      'return-name' => 'LxcEntry');
api(GET => '/nodes/{node}/lxc/{vmid}/config',           'lxc_get_config',      'param-name' => 'FixmeLxcGetConfig', 'return-name' => 'LxcConfig');
Schema2Rust::derive('LxcConfigNet' => 'Clone', 'PartialEq');
api(PUT => '/nodes/{node}/lxc/{vmid}/config', 'lxc_update_config', 'param-name' => 'UpdateLxcConfig');
api(GET => '/nodes/{node}/lxc/{vmid}/pending', 'lxc_get_pending', 'param-name' => 'FixmeLxcGetPending', 'output-type' => 'Vec<PendingConfigValue>');
api(POST => '/nodes/{node}/lxc/{vmid}/move_volume', 'lxc_move_volume', 'param-name' => 'LxcMoveVolume', 'output-type' => 'PveUpid');
api(PUT => '/nodes/{node}/lxc/{vmid}/resize', 'lxc_resize', 'param-name' => 'LxcResize', 'output-type' => 'PveUpid');
api(GET => '/nodes/{node}/lxc/{vmid}/status/current',   'lxc_get_status',      'return-name' => 'LxcStatus');
api(POST => '/nodes/{node}/lxc/{vmid}/status/start',    'start_lxc_async',     'output-type' => 'PveUpid', 'param-name' => 'StartLxc');
api(POST => '/nodes/{node}/lxc/{vmid}/status/stop',     'stop_lxc_async',      'output-type' => 'PveUpid', 'param-name' => 'StopLxc');
api(POST => '/nodes/{node}/lxc/{vmid}/status/shutdown', 'shutdown_lxc_async',  'output-type' => 'PveUpid', 'param-name' => 'ShutdownLxc');
Schema2Rust::derive('StartLxc' => 'Default');
Schema2Rust::derive('StopLxc' => 'Default');
Schema2Rust::derive('ShutdownLxc' => 'Default');
api(POST => '/nodes/{node}/lxc/{vmid}/migrate',        'migrate_lxc',         'output-type' => 'PveUpid', 'param-name' => 'MigrateLxc');
Schema2Rust::register_api_override('MigrateLxc', '/properties/bwlimit/default', undef);
api(POST => '/nodes/{node}/lxc/{vmid}/remote_migrate', 'remote_migrate_lxc',  'output-type' => 'PveUpid', 'param-name' => 'RemoteMigrateLxc');

Schema2Rust::register_enum_variant('ListNetworksType::OVSBridge' => 'OvsBridge');
Schema2Rust::register_enum_variant('ListNetworksType::OVSBond' => 'OvsBond');
Schema2Rust::register_enum_variant('ListNetworksType::OVSPort' => 'OvsPort');
Schema2Rust::register_enum_variant('ListNetworksType::OVSIntPort' => 'OvsIntPort');
Schema2Rust::register_enum_variant('NetworkInterfaceBondXmitHashPolicy::layer2+3' => 'Layer2_3');
Schema2Rust::register_enum_variant('NetworkInterfaceBondXmitHashPolicy::layer3+4' => 'Layer3_4');
Schema2Rust::register_enum_variant('NetworkInterfaceBondMode::802.3ad' => 'Ieee802_3ad');
Schema2Rust::register_enum_variant('NetworkInterfaceVlanProtocol::802.1ad' => 'Ieee802_1ad');
Schema2Rust::register_enum_variant('NetworkInterfaceVlanProtocol::802.1q' => 'Ieee802_1q');
api(GET => '/nodes/{node}/network', 'list_networks', 'return-name' => 'NetworkInterface');
Schema2Rust::derive('NetworkInterface' => 'Clone', 'PartialEq');

api(GET => '/nodes/{node}/storage', 'list_storages', 'return-name' => 'StorageInfo');
Schema2Rust::derive('StorageInfo' => 'Clone', 'PartialEq');
Schema2Rust::derive('StorageInfoFormats' => 'Clone', 'PartialEq');

api(GET => '/nodes/{node}/storage/{storage}/status', 'storage_status', 'return-name' => 'StorageStatus');

# FIXME: PVE9 introduced a new non optional property, but that does not
# exist in PVE8, so make it optional here for older PVEs to work
Schema2Rust::generate_struct(
    'NodeStatusMemory',
    {
        type => 'object',
        properties => {
            'available' => {
                type => 'integer',
                description => 'The available memory in bytes.',
                optional => 1,
            },
            'free' => {
                type => 'integer',
                description => 'The free memory in bytes.',
            },
            'total' => {
                type => 'integer',
                description => 'The total memory in bytes.',
            },
            'used' => {
                type => 'integer',
                description => 'The used memory in bytes.',
            },
        },
    },
    {},
    {},
);
api(GET => '/nodes/{node}/status', 'node_status', 'return-name' => 'NodeStatus');

api(GET => '/nodes/{node}/capabilities/qemu/cpu', 'qemu_cpu_capabilities', 'output-type' => 'Vec<QemuCpuModel>');


Schema2Rust::register_api_override('ClusterMetrics', '/properties/data/items', { type => "ClusterMetricsData"});
api(GET => '/cluster/metrics/export', 'cluster_metrics_export', 'return-name' => 'ClusterMetrics');

Schema2Rust::register_api_extensions('ClusterJoinInfoNodelist', {
    '/properties/pve_addr' => { description => sq("FIXME: Missing description in PVE.") },
    '/properties/pve_fp' => { description => sq("FIXME: Missing description in PVE.") },
    '/properties/quorum_votes' => { description => sq("FIXME: Missing description in PVE.") },
});
Schema2Rust::register_api_extensions('ClusterJoinInfo', {
    '/properties/config_digest' => { description => sq("FIXME: Missing description in PVE.") },
    '/properties/nodelist' => { description => sq("FIXME: Missing description in PVE.") },
});
api(GET => '/cluster/config/join', 'cluster_config_join', 'return-name' => 'ClusterJoinInfo');

# cluster status info
Schema2Rust::register_api_extensions('ClusterNodeStatus', {
    '/properties/id' => { description => sq("FIXME: Missing description in PVE.") },
    '/properties/name' => { description => sq("FIXME: Missing description in PVE.") },
});
api(GET => '/cluster/status', 'cluster_status', 'return-name' => 'ClusterNodeStatus');

# api(GET => '/storage', 'list_storages', 'return-name' => 'StorageList');
Schema2Rust::register_api_extensions('ListRealm', {
    '/properties/realm' => { description => sq("FIXME: Missing description in PVE.") },
    '/properties/type' => { description => sq("FIXME: Missing description in PVE.") },
});
api(GET => '/access/domains', 'list_domains', 'return-name' => 'ListRealm');
Schema2Rust::derive('ListRealm' => 'Clone', 'PartialEq');
# api(GET => '/access/groups', 'list_groups', 'return-name' => 'ListGroups');
# api(GET => '/access/groups/{groupid}', 'get_group', 'return-name' => 'Group');
# api(GET => '/access/users', 'list_users', 'return-name' => 'ListUsers');
# api(GET => '/access/users/{userid}', 'get_user', 'return-name' => 'User');
api(POST => '/access/users/{userid}/token/{tokenid}', 'create_token', 'param-name' => 'CreateToken');
Schema2Rust::derive('CreateToken' => 'Default');

api(GET => '/nodes/{node}/apt/update', 'list_available_updates', 'return-name' => 'AptUpdateInfo');
api(POST => '/nodes/{node}/apt/update', 'update_apt_database', 'output-type' => 'PveUpid', 'param-name' => 'AptUpdateParams');
api(GET => '/nodes/{node}/apt/changelog', 'get_package_changelog', 'output-type' => 'String');
api(GET => '/nodes/{node}/apt/versions', 'get_package_versions', 'return-name' => 'InstalledPackage');
api(GET => '/nodes/{node}/apt/repositories', 'get_apt_repositories', 'output-type' => 'APTRepositoriesResult');

Schema2Rust::generate_enum('FwConntrackHelper', {
    type => 'string',
    description => "Firewall conntrack helper.",
    enum => ['amanda', 'ftp', 'irc', 'netbios-ns', 'pptp', 'sane', 'sip', 'snmp', 'tftp'],
});

Schema2Rust::register_enum_variant('FirewallLogLevel::err' => 'Error');
Schema2Rust::generate_enum('FirewallLogLevel', {
    type => 'string',
    description => "Firewall log levels.",
    enum => ['emerg', 'alert', 'crit', 'err', 'warning', 'notice', 'info', 'debug', 'nolog'],
    default => 'nolog',
});

Schema2Rust::generate_enum('FirewallIOPolicy', {
    type => 'string',
    description => "Firewall IO policies.",
    enum => ['ACCEPT', 'DROP', 'REJECT'],
});

Schema2Rust::generate_enum('FirewallFWPolicy', {
    type => 'string',
    description => "Firewall IO policies.",
    enum => ['ACCEPT', 'DROP'],
    default => 'ACCEPT',
});

Schema2Rust::register_format('pve-fw-conntrack-helper' => {
    type => 'FwConntrackHelper',
    kind => 'array',
});

# options
# FIXME: to use a better return value than `Value`, we first must fix the return schema there
api(GET => '/cluster/options', 'cluster_options', 'output-type' => 'serde_json::Value');

# firewall options
api(GET => '/cluster/firewall/options', 'cluster_firewall_options', 'return-name' => 'ClusterFirewallOptions');
api(PUT => '/cluster/firewall/options', 'set_cluster_firewall_options', 'param-name' => 'UpdateClusterFirewallOptions');

api(GET => '/nodes/{node}/firewall/options', 'node_firewall_options', 'return-name' => 'NodeFirewallOptions');
api(PUT => '/nodes/{node}/firewall/options', 'set_node_firewall_options', 'param-name' => 'UpdateNodeFirewallOptions');

api(GET => '/nodes/{node}/lxc/{vmid}/firewall/options', 'lxc_firewall_options', 'return-name' => 'GuestFirewallOptions');
api(PUT => '/nodes/{node}/lxc/{vmid}/firewall/options', 'set_lxc_firewall_options', 'param-name' => 'UpdateGuestFirewallOptions');
api(GET => '/nodes/{node}/qemu/{vmid}/firewall/options', 'qemu_firewall_options', 'return-name' => 'GuestFirewallOptions');
api(PUT => '/nodes/{node}/qemu/{vmid}/firewall/options', 'set_qemu_firewall_options', 'param-name' => 'UpdateGuestFirewallOptions');

# firewall rules
api(GET => '/cluster/firewall/rules', 'list_cluster_firewall_rules', 'return-name' => 'ListFirewallRules');

api(GET => '/nodes/{node}/firewall/rules', 'list_node_firewall_rules', 'return-name' => 'ListFirewallRules');

api(GET => '/nodes/{node}/lxc/{vmid}/firewall/rules', 'list_lxc_firewall_rules', 'return-name' => 'ListFirewallRules');
api(GET => '/nodes/{node}/qemu/{vmid}/firewall/rules', 'list_qemu_firewall_rules', 'return-name' => 'ListFirewallRules');

# firewall log
api(GET => '/nodes/{node}/firewall/log', 'node_firewall_log', 'return-name' => 'TaskLogLine', attribs => 1);
api(GET => '/nodes/{node}/lxc/{vmid}/firewall/log', 'lxc_firewall_log', 'return-name' => 'TaskLogLine', attribs => 1);
api(GET => '/nodes/{node}/qemu/{vmid}/firewall/log', 'qemu_firewall_log', 'return-name' => 'TaskLogLine', attribs => 1);

# firewall aliases
api(GET => '/cluster/firewall/aliases', 'cluster_firewall_aliases', 'return-name' => 'FirewallAlias');
api(POST => '/cluster/firewall/aliases', 'create_cluster_firewall_alias', 'param-name' => 'CreateFirewallAlias');
api(PUT => '/cluster/firewall/aliases/{name}', 'update_cluster_firewall_alias', 'param-name' => 'UpdateFirewallAlias');
api(DELETE => '/cluster/firewall/aliases/{name}', 'delete_cluster_firewall_alias', 'param-name' => 'DeleteFirewallAlias');

api(GET => '/nodes/{node}/lxc/{vmid}/firewall/aliases', 'lxc_firewall_aliases', 'return-name' => 'FirewallAlias');
api(POST => '/nodes/{node}/lxc/{vmid}/firewall/aliases', 'create_lxc_firewall_alias', 'param-name' => 'CreateFirewallAlias');
api(PUT => '/nodes/{node}/lxc/{vmid}/firewall/aliases/{name}', 'update_lxc_firewall_alias', 'param-name' => 'UpdateFirewallAlias');
api(DELETE => '/nodes/{node}/lxc/{vmid}/firewall/aliases/{name}', 'delete_lxc_firewall_alias', 'param-name' => 'DeleteFirewallAlias');

api(GET => '/nodes/{node}/qemu/{vmid}/firewall/aliases', 'qemu_firewall_aliases', 'return-name' => 'FirewallAlias');
api(POST => '/nodes/{node}/qemu/{vmid}/firewall/aliases', 'create_qemu_firewall_alias', 'param-name' => 'CreateFirewallAlias');
api(PUT => '/nodes/{node}/qemu/{vmid}/firewall/aliases/{name}', 'update_qemu_firewall_alias', 'param-name' => 'UpdateFirewallAlias');
api(DELETE => '/nodes/{node}/qemu/{vmid}/firewall/aliases/{name}', 'delete_qemu_firewall_alias', 'param-name' => 'DeleteFirewallAlias');

# /cluster/firewall/ipset
api(GET => '/cluster/firewall/ipset', 'cluster_firewall_ipset_list', 'return-name' => 'FirewallIpSetListItem');
api(GET => '/cluster/firewall/ipset/{name}', 'cluster_firewall_ipset', 'return-name' => 'FirewallIpSet');
api(DELETE => '/cluster/firewall/ipset/{name}', 'delete_cluster_firewall_ipset', 'param-name' => 'DeleteFirewallIpSet');
api(POST => '/cluster/firewall/ipset/{name}', 'create_cluster_firewall_ipset_entry', 'param-name' => 'CreateFirewallIpSetEntry');
api(GET => '/cluster/firewall/ipset/{name}/{cidr}', 'cluster_firewall_ipset_entry', 'return-name' => 'FirewallIpSetEntry');
api(PUT => '/cluster/firewall/ipset/{name}/{cidr}', 'update_cluster_firewall_ipset_entry', 'param-name' => 'UpdateFirewallIpSetEntry');
api(DELETE => '/cluster/firewall/ipset/{name}/{cidr}', 'delete_cluster_firewall_ipset_entry', 'param-name' => 'DeleteFirewallIpSetEntry');

# /nodes/{node}/lxc/{vmid}/firewall/ipset
api(GET => '/nodes/{node}/lxc/{vmid}/firewall/ipset', 'lxc_firewall_ipset_list', 'return-name' => 'FirewallIpSetListItem');
api(GET => '/nodes/{node}/lxc/{vmid}/firewall/ipset/{name}', 'lxc_firewall_ipset', 'return-name' => 'FirewallIpSet');
api(DELETE => '/nodes/{node}/lxc/{vmid}/firewall/ipset/{name}', 'delete_lxc_firewall_ipset', 'param-name' => 'DeleteFirewallIpSet');
api(POST => '/nodes/{node}/lxc/{vmid}/firewall/ipset/{name}', 'create_lxc_firewall_ipset_entry', 'param-name' => 'CreateFirewallIpSetEntry');
api(GET => '/nodes/{node}/lxc/{vmid}/firewall/ipset/{name}/{cidr}', 'lxc_firewall_ipset_entry', 'return-name' => 'FirewallIpSetEntry');
api(PUT => '/nodes/{node}/lxc/{vmid}/firewall/ipset/{name}/{cidr}', 'update_lxc_firewall_ipset_entry', 'param-name' => 'UpdateFirewallIpSetEntry');
api(DELETE => '/nodes/{node}/lxc/{vmid}/firewall/ipset/{name}/{cidr}', 'delete_lxc_firewall_ipset_entry', 'param-name' => 'DeleteFirewallIpSetEntry');

# /nodes/{node}/qemu/{vmid}/firewall/ipset
api(GET => '/nodes/{node}/qemu/{vmid}/firewall/ipset', 'qemu_firewall_ipset_list', 'return-name' => 'FirewallIpSetListItem');
api(GET => '/nodes/{node}/qemu/{vmid}/firewall/ipset/{name}', 'qemu_firewall_ipset', 'return-name' => 'FirewallIpSet');
api(DELETE => '/nodes/{node}/qemu/{vmid}/firewall/ipset/{name}', 'delete_qemu_firewall_ipset', 'param-name' => 'DeleteFirewallIpSet');
api(POST => '/nodes/{node}/qemu/{vmid}/firewall/ipset/{name}', 'create_qemu_firewall_ipset_entry', 'param-name' => 'CreateFirewallIpSetEntry');
api(GET => '/nodes/{node}/qemu/{vmid}/firewall/ipset/{name}/{cidr}', 'qemu_firewall_ipset_entry', 'return-name' => 'FirewallIpSetEntry');
api(PUT => '/nodes/{node}/qemu/{vmid}/firewall/ipset/{name}/{cidr}', 'update_qemu_firewall_ipset_entry', 'param-name' => 'UpdateFirewallIpSetEntry');
api(DELETE => '/nodes/{node}/qemu/{vmid}/firewall/ipset/{name}/{cidr}', 'delete_qemu_firewall_ipset_entry', 'param-name' => 'DeleteFirewallIpSetEntry');

# firewall macros
api(GET => '/cluster/firewall/macros', 'cluster_firewall_macros', 'return-name' => 'FirewallMacro');

# firewall refs
api(GET => '/cluster/firewall/refs', 'cluster_firewall_refs', 'return-name' => 'FirewallRef');
api(GET => '/nodes/{node}/lxc/{vmid}/firewall/refs', 'lxc_firewall_refs', 'return-name' => 'FirewallRef');
api(GET => '/nodes/{node}/qemu/{vmid}/firewall/refs', 'qemu_firewall_refs', 'return-name' => 'FirewallRef');

Schema2Rust::register_api_extensions('UpdateClusterFirewallOptions', {
    '/properties/comment' => { description => sq("Descriptive comment") },
});
Schema2Rust::register_api_extensions('CreateFirewallAlias', {
    '/properties/comment' => { description => sq("Descriptive comment") },
});
Schema2Rust::register_api_extensions('UpdateFirewallAlias', {
    '/properties/comment' => { description => sq("Descriptive comment") },
});
Schema2Rust::register_api_extensions('FirewallAlias', {
    '/properties/name' => { description => sq("Alias name") },
    '/properties/comment' => { description => sq("Descriptive comment") },
    '/properties/cidr' => { description => sq("CIDR address") },
});
Schema2Rust::register_api_extensions('CreateFirewallIpSetEntry', {
    '/properties/comment' => { description => sq("Descriptive comment") },
    '/properties/cidr' => { description => sq("CIDR address") },
    '/properties/nomatch' => { description => sq("Inversed matching") },
});

Schema2Rust::register_api_extensions('FirewallIpSetListItem', {
    '/properties/comment' => { description => sq("Descriptive comment") },
    '/properties/cidr' => { description => sq("CIDR address") },
    '/properties/nomatch' => { description => sq("Inversed matching") },
});

Schema2Rust::register_api_extensions('FirewallIpSet', {
    '/properties/comment' => { description => sq("Descriptive comment") },
    '/properties/cidr' => { description => sq("CIDR address") },
    '/properties/nomatch' => { description => sq("Inversed matching") },
});

Schema2Rust::register_api_extensions('UpdateFirewallIpSetEntry', {
    '/properties/comment' => { description => sq("Descriptive comment") },
    '/properties/cidr' => { description => sq("CIDR address") },
    '/properties/nomatch' => { description => sq("Inversed matching") },
});

Schema2Rust::register_api_extensions('FirewallRef', {
    '/properties/comment' => { description => sq("Descriptive comment") },
    '/properties/name' => { description => sq("The name of the alias or ipset.") },
    '/properties/ref' => { description => sq("The reference string used in firewall rules.") },
    '/properties/scope' => { description => sq("The scope of the reference (e.g., SDN).") },
});

Schema2Rust::derive('ListFirewallRules' => 'Clone', 'PartialEq');
Schema2Rust::derive('FirewallAlias' => 'Clone', 'PartialEq');
Schema2Rust::derive('FirewallMacro' => 'Clone', 'PartialEq');
Schema2Rust::derive('FirewallRef' => 'Clone', 'PartialEq');
Schema2Rust::derive('FirewallIpSetListItem' => 'Clone', 'PartialEq');

Schema2Rust::generate_enum('SdnObjectState', {
    type => 'string',
    description => "The state of an SDN object.",
    enum => ['new', 'deleted', 'changed'],
});

api(GET => '/cluster/sdn/zones', 'list_zones', 'return-name' => 'SdnZone');
Schema2Rust::derive('SdnZone' => 'Clone', 'PartialEq');
Schema2Rust::derive('SdnZonePending' => 'Clone', 'PartialEq');
api(POST => '/cluster/sdn/zones', 'create_zone', 'param-name' => 'CreateZone');
Schema2Rust::derive('CreateZone' => 'Clone', 'PartialEq');
api(GET => '/nodes/{node}/sdn/zones/{zone}/ip-vrf', 'get_zone_ip_vrf', 'return-name' => 'SdnZoneIpVrf');
Schema2Rust::derive('SdnZoneIpVrf' => 'Clone', 'PartialEq');

api(GET => '/cluster/sdn/controllers', 'list_controllers', 'return-name' => 'SdnController');
Schema2Rust::derive('SdnController' => 'Clone', 'PartialEq');
Schema2Rust::derive('SdnControllerPending' => 'Clone', 'PartialEq');
api(POST => '/cluster/sdn/controllers', 'create_controller', 'param-name' => 'CreateController');
Schema2Rust::derive('CreateController' => 'Clone', 'PartialEq');

api(GET => '/cluster/sdn/vnets', 'list_vnets', 'return-name' => 'SdnVnet');
Schema2Rust::derive('SdnVnet' => 'Clone', 'PartialEq');
Schema2Rust::derive('SdnVnetPending' => 'Clone', 'PartialEq');
api(POST => '/cluster/sdn/vnets', 'create_vnet', 'param-name' => 'CreateVnet');
Schema2Rust::derive('CreateVnet' => 'Clone', 'PartialEq');
api(GET => '/nodes/{node}/sdn/vnets/{vnet}/mac-vrf', 'get_vnet_mac_vrf', 'return-name' => 'SdnVnetMacVrf');
Schema2Rust::derive('SdnVnetMacVrf' => 'Clone', 'PartialEq');

api(POST => '/cluster/sdn/lock', 'acquire_sdn_lock', 'param-name' => 'CreateSdnLock', 'output-type' => 'String');
api(DELETE => '/cluster/sdn/lock', 'release_sdn_lock', 'param-name' => 'ReleaseSdnLock');

api(PUT => '/cluster/sdn', 'sdn_apply', 'param-name' => 'ReloadSdn', 'output-type' => 'PveUpid');
api(POST => '/cluster/sdn/rollback', 'rollback_sdn_changes', 'param-name' => 'RollbackSdn');

api(POST => '/nodes/{node}/termproxy', 'node_shell_termproxy', 'param-name' => 'NodeShellTermproxy', 'return-name' => 'NodeShellTicket');

# NOW DUMP THE CODE:
#
# We generate one file for API types, and one for API method calls.

my $type_out_file = "$output_dir/types.rs";
my $code_out_file = "$output_dir/code.rs";

# Redirect code generation through rustfmt:
open(my $type_fh, '>', $type_out_file) or die "failed to open '$type_out_file': $!\n";
my $type_pid = open2(
    '>&'.fileno($type_fh),
    my $type_pipe,
    #'cat',
    'rustfmt', '--edition=2021', '--config', 'wrap_comments=true'
);
open(my $code_fh, '>', $code_out_file) or die "failed to open '$code_out_file': $!\n";
my $code_pid = open2(
    '>&'.fileno($code_fh),
    my $code_pipe,
    #'cat',
    'rustfmt', '--edition=2021', '--config', 'wrap_comments=true'
);
close($type_fh);
close($code_fh);

# Create .rs files:
print {$code_pipe} "/// PVE API client\n";
print {$code_pipe} "/// Note that the following API URLs are not handled currently:\n";
print {$code_pipe} "///\n";
print {$code_pipe} "/// ```text\n";
my $unused = Schema2Rust::get_unused_paths();
for my $path (sort keys $unused->%*) {
    print {$code_pipe} "/// - $path\n";
}
print {$code_pipe} "/// ```\n";

# Schema2Rust::dump();
Schema2Rust::print_types($type_pipe);
Schema2Rust::print_trait($code_pipe);
print {$code_pipe} "\n";
Schema2Rust::print_implementation($code_pipe);
$type_pipe->flush();
$code_pipe->flush();
close($type_pipe);
close($code_pipe);

# Wait for formatters to finish:
do {} while $type_pid != waitpid($type_pid, 0);
do {} while $code_pid != waitpid($code_pid, 0);
