# Generated API types for perl based proxmox products

This is meant to contain the API types converted from perl to rust for 3 main
reasons:

1. To use them in wasm based UIs.
2. To use them in PDM.
3. To more visibly track/record (incompatible) schema changes in git.

Currently the implementation only deals with PVE and the generator code lives in
the `pve-api-types/` subdirectory. Eventually, changes will be made to also
build bindings for PMG.

# Code generation

Code generation consists of several parts:

## `pve-api-types/generator-lib`

This contains the "heavy lifting" code dealing with analyzing a perl schema and
producing not only rust *types*, but also the corresponding `#[api]` annotations
for it. This also contains some magic to deal with our "quirks" such as the fact
that arrays are generated by adding a bunch of `name${index}` entries into an
object at the top level instead of nesting an array inside an object.

This code has unfortunately "grown organically" a lot as more and more cases
have been uncovered by generating data for more API calls and is probably in
need of a cleanup...

## `pve-api.json`

This is a PVE API dump with references preserved. This is generated from
*installed* PVE packages and should not be modified manually.
This is the source used by the generator to produce types and code.

## Using the `generator-lib`

The actual generation is triggered by `pve-api-types/generate.pl`.

1. This file loads the entire API from `pve-api.json`.
2. Then it calls `Schema2Rust::init_api()` to "set up" the API "router".
3. Finally its `api()` sub is used to cause an API path to be turned into a
   *method* of the pve `Client` type in rust.

However, before this can happen, we need to deal with all the things we *cannot*
convert automatically. Specifically, a lot of "formats" which use perl *code*
need to be "registered" first via `Schema2Rust::register_format`.

But first, a note on the generated type names and how to cause an API endpoint
to gain a method in the generated client:

### Type names

While analyzing a schema, all rust types will receive a name. The name mostly
consists of some kind of starting point (coming from an `api()` invocation),
with the "path" to the value added in a hopefully rust-typical manner.

For instance

```
api(GET => '/nodes/{node}/qemu/{vmid}/config', 'qemu_get_config', 'param-name' => 'FixmeQemuGetConfig', 'return-name' => 'QemuConfig');
```

The `param-name` provides a starting point for the *parameter* `struct` (which
does not exist for this call, hence the name, to point out something unexpected
changed when it appears in the code).

The `return-name` provides a starting point for the *return type*.

This generates a method named `qemu_get_config` from the given API path.
Its *return* schema will end up creating the rust type named:

    struct QemuConfig { ... }

Since `QemuConfig` has a member named `efidisk0` which is a "property string", a
`struct` for this property string will be added named:

    struct QemuConfigEfidisk0 { ... }

In there, an `enum` is encountered for the `efitype` property. For this enum, a
rust `enum` will be created named:

    enum QemuConfigEfidisk0Efitype {}

This `enum` has 2 problematic variants: `2m` and `4m`. We cannot type

    enum QemuConfigEfidisk0Efitype { 2m, 4m }

as this is not valid rust syntax...

### Dealing with rust-unfriendly `enum` variants.

To fix this, we use the enum's type name and original variant name and create a
mapping:

```
Schema2Rust::register_enum_variant('QemuConfigEfidisk0Efitype::2m' => 'Mb2');
Schema2Rust::register_enum_variant('QemuConfigEfidisk0Efitype::4m' => 'Mb4');
```

### Type dedupliation vs type names - ORDER MATTERS

Note that the generator lib will produce a kind of "signature" data for a
schema, in an attempt to "deduplicate" types which get reused *or copied* in
perl (`'some-property' => %$some_schema_variable`).

This means that the *order* in which the `api()` invocations are made *may
affect the names the API types receive*.

So to provide a stable rust API, the code in `generator.pl` *also needs to have
a stable order*.

### Adding format verifiers:

A format can have the following forms:
- `{ code => "function_name" }`
  This translates to `ApiStringFormat::VerifyFn(function_name)`.
- `{ regex => "a_regex" }`
  Note that there may be subtle differences in how rust's `regex` crate and
  perl's regular expressions work, which is why this format must be registered
  manually and is not automatically converted.
  This translates to adding a `const_regex! {}` entry and using
  `ApiStringFormat::Pattern(const_regex_name)`.
- `{ unchecked => 1 }`
  Unchecked types. Use sparingly. This is probably mostly for "TODO" entry.
- `{ type => "SomeType" }`
  Use a particular rust type instead of a `String` and trust that its
  deserialization takes care of validation.
- `{ property_string => "SomeApiType" }`
  Use `ApiStringFormat::PropertyString(&SomeApiType::API_SCHEMA)`.

### "Extending" properties

There's a `Schema2Rust::register_api_extension()` sub which can attach
additional data to the collected API type information.

This can be used to "fixup" missing `descriptions` or other simple
fields.

HOWEVER: This is applied *while writing out the #[api] annotation* and will
*NOT* be taken into account during the "analyzing" phase of the schema.

It is therefore *NOT* suitable or meant to write out complex additional types.
And won't work like that - on purpose! Go fix the perl code instead.

### "Overriding" properties

Similar to the "extensions" we also have *overrides*. The same caveats apply.
These are mostly used to fix up "default" values which don't specify actual
values but rather textual descriptions (since in perl, the defaults are only
used for documentation purposes, while in rust we *try* to force them to be
valid values).

Example:

```
# We have a textual description of the default value in there, just pick the cgroupv2 one:
Schema2Rust::register_api_override('QemuConfig', '/properties/cpuunits/default', 1024);
```

### Generating *new* enums.

Sometimes we do want/need to add additional types, but this should be the
exception, not the rule. Here's an example (which may as well be fixed in the
perl schema, but for the sake of testing that additional enums worked...)

```
# pve-storage-content uses verify_
my $storage_content_types = [sort keys PVE::Storage::Plugin::valid_content_types('dir')->%*];
Schema2Rust::generate_enum('StorageContent', {
    type => 'string',
    description => 'Storage content type.',
    enum => $storage_content_types,
});
```

### Adding `#[derive]` attributes to generated types.

This can be achieved with the `derive()` sub like so.

NOTE: The type needs to have already been encountered, so this should happen
*after* the corresponding `api()` call generating the type name.
(This way you'll immediately see an error if such a type vanishes from the code...)

```
Schema2Rust::derive('ListTasks' => 'Default');
```
